import { Component, OnDestroy, OnInit, ChangeDetectorRef } from '@angular/core';
import { filter, finalize, map, Subscription } from 'rxjs';
import { TranslateService } from '@ngx-translate/core';
import { OnlineStatusService, OnlineStatusType } from 'ngx-online-status';
import { SwUpdate, VersionReadyEvent } from '@angular/service-worker';
import { ConfirmEventType, ConfirmationService } from 'primeng/api';
import {
  NavigationCancel,
  NavigationEnd,
  NavigationError,
  NavigationStart,
  Router,
  Event as RouterEvent,
} from '@angular/router';
import { scrollTop } from '@helpers/window.helper';
import { Event as NavigationEvent } from '@angular/router';
import { LanguagesService } from '@services/languages.service';
import { TranslateHelper } from '@helpers/translate.helper';
import { CommonService } from '@services/common.service';
import { response } from '@interfaces/response';
import { Address } from '@enums/address.enum';
import { MenuStateStore } from '@stores/loadMenu.store';
import { LogingoutStore } from '@stores/isLogingout.store';
@Component({
  selector: 'app-root',
  templateUrl: './app.component.html',
  styleUrls: ['./app.component.scss'],
})
export class AppComponent implements OnInit, OnDestroy {
  status: OnlineStatusType;
  loadingPage = false;
  isLogout: boolean;
  subscriptions = new Subscription();
  menuState: boolean;
  isCloseAlarmer: boolean = true;
  constructor(
    private _translateService: TranslateService,
    private onlineStatusService: OnlineStatusService,
    private swUpdate: SwUpdate,
    private _router: Router,
    private confirmationService: ConfirmationService,
    private _language: LanguagesService,
    private _translateHelper: TranslateHelper,
    private _commonService: CommonService,
    private _menuStateStore: MenuStateStore,
    private _logoutStore: LogingoutStore,
    private cdr: ChangeDetectorRef
  ) {
    this._router.events.subscribe((e: RouterEvent) => {
      this.navigationInterceptor(e);
    });
    this.onlineStatusService.status.subscribe((status: OnlineStatusType) => {
      this.status = status;
    });

    this._menuStateStore.menu.subscribe((res) => {
      this.menuState = res;
    });

    this._logoutStore.isLogout.subscribe((res) => {
      this.isLogout = res;
    });
  }
  closeAlarmer() {
    this.isCloseAlarmer = false;
  }
  navigationInterceptor(event: RouterEvent): void {
    if (event instanceof NavigationStart) {
      this.loadingPage = true;
      this.cdr.detectChanges();
    }
    if (event instanceof NavigationEnd) {
      this.loadingPage = false;
      this.cdr.detectChanges();
    }

    if (event instanceof NavigationCancel) {
      this.loadingPage = false;
      this.cdr.detectChanges();
    }
    if (event instanceof NavigationError) {
      this.loadingPage = false;
      this.cdr.detectChanges();
    }
  }

  onActivate(event) {
    scrollTop();
  }

  confirmUpdate() {
    this.confirmationService.confirm({
      message:
        'نسخه جدیدی از برنامه در دسترس قرار گرفته است ، آیا مایل به بروز رسانی هستید؟',
      header: 'بروز رسانی برنامه',
      icon: 'pi pi-info-circle',
      accept: () => {
        this.updateVersion();
      },
      reject: (type: ConfirmEventType) => {
        switch (type) {
          case ConfirmEventType.REJECT:
            break;
          case ConfirmEventType.CANCEL:
            break;
        }
      },
      key: 'dialogUpdate',
    });
  }

  ngOnInit(): void {
    this.swUpdate.available.subscribe((evt) => {
      this.updateVersion();
    });
    if (this.swUpdate.isEnabled) {
      this.swUpdate.versionUpdates.pipe(
        filter(
          (evt: any): evt is VersionReadyEvent => evt.type === 'VERSION_READY'
        ),
        map((evt: any) => {
          this.updateVersion();
        })
      );
    }
    this.setTranslateFromUrl();
  }

  updateVersion(): void {
    window.location.reload();
  }

  ngOnDestroy() {
    this.subscriptions.unsubscribe();
  }
  updateLanguage(key) {
    this.subscriptions.add(
      this._commonService
        .put(Address.LanguageUpdate, { key: key }, false)
        .pipe(finalize(() => {}))
        .subscribe(
          (response: response) => {},
          (error) => {}
        )
    );
  }
  setTranslateFromUrl() {
    let languageList = this._language.languages;
    this._router.events.subscribe((event: NavigationEvent): void => {
      if (event instanceof NavigationEnd) {
        let urlSegments = event.url.split('/');
        if (
          urlSegments != undefined &&
          urlSegments != null &&
          urlSegments.length > 0
        ) {
          if (
            languageList.filter(
              (a) => a.key == urlSegments[urlSegments.length - 1]
            ) != undefined &&
            languageList.filter(
              (a) => a.key == urlSegments[urlSegments.length - 1]
            ) != null &&
            languageList.filter(
              (a) => a.key == urlSegments[urlSegments.length - 1]
            ).length > 0
          ) {
            this.updateLanguage(urlSegments[urlSegments.length - 1]);
            this._translateService.setDefaultLang(
              urlSegments[urlSegments.length - 1]
            );
            this._translateHelper.rtlLtr(urlSegments[urlSegments.length - 1]);
          } else {
            this._translateService.setDefaultLang('fa');
            this._translateHelper.rtlLtr('fa');
          }
        } else {
          this._translateService.setDefaultLang('fa');
          this._translateHelper.rtlLtr('fa');
        }
      }
    });
  }
}
