import { Component, OnDestroy, ViewChild } from '@angular/core';
import { CommonService } from '@services/common.service';
import { Subscription } from 'rxjs';
import { finalize, filter, pairwise, tap } from 'rxjs/operators';
import { response } from '@interfaces/response';
import { isInteger, isValidEmail } from '@helpers/validator.helper';
import { Router, RoutesRecognized } from '@angular/router';
import { Address } from '@enums/address.enum';
import { CaptchaComponent } from '@components/captcha/captcha.component';
import { setSessionStorageItem } from '@helpers/sessionStorage.helper';
import { ColumnKey } from '@enums/columnKey.enum';
import { LoginModel } from 'src/app/core/params/auth/loginModel';
import { AlertHelper } from '@helpers/alert.helper';
import { RedirectorHelper } from '@helpers/redirector.helper';
import { AuthStore } from '@stores/AuthSetting.store';
import { UsernameStore } from '@stores/loginUsernameStore.store';
import { toEnglishDigits } from '@helpers/convertToEnglishDigit.helper';

@Component({
  selector: 'form-login',
  templateUrl: './formLogin.component.html',
  styleUrls: ['./formLogin.component.scss'],
})
export class FormLoginComponent implements OnDestroy {
  isLoading: boolean = false;
  subscriptions = new Subscription();
  isNumber: boolean = true;
  model: LoginModel = new LoginModel();
  prevUrl: string;
  currentUrl: string;

  isActiveCaptcha: boolean;
  @ViewChild('captcha') captcha: CaptchaComponent;
  @ViewChild('loginForm') loginForm: any;
  constructor(
    private _commonService: CommonService,
    private _router: Router,
    private _alert: AlertHelper,
    public _redirectorHelper: RedirectorHelper,
    private _authSettingStore: AuthStore,
    private _usernameStore: UsernameStore
  ) {
    this.subscriptions = this._router.events
      .pipe(
        filter((evt: any) => evt instanceof RoutesRecognized),
        tap((event) => {}),
        pairwise()
      )
      .subscribe((events: RoutesRecognized[]) => {});

    this._authSettingStore.isActiveCaptcha.subscribe((res) => {
      if (res != null) {
        this.isActiveCaptcha = res;
      }
    });

    this._authSettingStore.isActiveCaptchaBehavior.subscribe((res) => {
      if (res != null) {
        this.isActiveCaptcha = res;
      }
    });

    this._usernameStore.username.subscribe((res) => {
      if (res != null) {
        this.model.address = res;
      }
    });

    this._usernameStore.usernameBehavior.subscribe((res) => {
      if (res != null) {
        this.model.address = res;
      }
    });
  }

  ngOnDestroy(): void {
    this.subscriptions.unsubscribe();
    this.loginForm.resetForm();
  }
  getSubmitMethod(): any {
    return {
      callSubmitMethod: () => {
        this.submit();
      },
    };
  }
  submit() {
    if (isInteger(this.model.address)) this.model.typeVerify = 101;
    else {
      if (!isValidEmail(this.model.address)) {
        this._alert.warning('ایمیل وارد شده صحیح نمی باشد');
        return;
      }
      this.model.typeVerify = 102;
    }
    if (
      (this.captcha &&
        this.captcha.captchaInfo &&
        this.captcha.captchaInfo.captchaId &&
        this.captcha.captchaInfo.captchaCode) || !this.isActiveCaptcha
    ) {
      if (this.isActiveCaptcha) {
        const { captchaCode, captchaId } = this.captcha.captchaInfo;
        this.model.captchaCode = captchaCode;
        this.model.captchaCode = toEnglishDigits(this.model.captchaCode)
        this.model.captchaId = captchaId;
      } else {
        this.model.captchaCode = null
        this.model.captchaId = null
      }
      this.isLoading = true;

      this.subscriptions.add(
        this._commonService
          .post(Address.Send, this.model, false)
          .pipe(
            finalize(() => {
              this.isLoading = false;
              this.captcha.handleRefreshCaptcha();
            })
          )
          .subscribe(
            (response: response) => {
              setSessionStorageItem(ColumnKey.Address, response.data.address);
              setSessionStorageItem(ColumnKey.TokenVerify, response.data.token);
              setSessionStorageItem(
                ColumnKey.TypeVerify,
                response.data.typeVerify
              );
              this._redirectorHelper.routerHandlerAuth('verify');
            },
            (error) => {}
          )
      );
    }
  }

}

