import { Component, OnDestroy, ViewChild } from '@angular/core';
import { CommonService } from '@services/common.service';
import { Subscription } from 'rxjs';
import { finalize, filter, pairwise, tap } from 'rxjs/operators';
import { response } from '@interfaces/response';
import { Router, RoutesRecognized } from '@angular/router';
import { Address } from '@enums/address.enum';
import { ColumnKey } from '@enums/columnKey.enum';
import { CaptchaComponent } from '@components/captcha/captcha.component';
import { LoginByUserNameModel } from 'src/app/core/params/auth/loginByUserName';
import { setStorageItemCallback } from '@helpers/localstorage.helper';
import { RedirectorHelper } from '@helpers/redirector.helper';
import { AuthStore } from '@stores/AuthSetting.store';
import { UsernameStore } from '@stores/loginUsernameStore.store';
import { isValidEmail, isValidMobile } from '@helpers/validator.helper';
import { AlertHelper } from '@helpers/alert.helper';
import { toEnglishDigits } from '@helpers/convertToEnglishDigit.helper';

@Component({
  selector: 'form-login-by-username',
  templateUrl: './formLoginByUserName.component.html',
  styleUrls: ['./formLoginByUserName.component.scss'],
})
export class FormLoginByUserNameComponent implements OnDestroy {
  isLoading: boolean = false;

  subscriptions = new Subscription();

  isNumber: boolean = true;

  model: LoginByUserNameModel = new LoginByUserNameModel();

  isActiveCaptcha: boolean;
  @ViewChild('captcha') captcha: CaptchaComponent;

  constructor(
    private _commonService: CommonService,
    private _router: Router,
    public _redirectorHelper: RedirectorHelper,
    private _authSettingStore: AuthStore,
    private _usernameStore: UsernameStore,
    private _alert: AlertHelper,
  ) {
    this.subscriptions = this._router.events
      .pipe(
        filter((evt: any) => evt instanceof RoutesRecognized),
        tap((event) => { }),
        pairwise()
      )
      .subscribe((events: RoutesRecognized[]) => { });

    this._authSettingStore.isActiveCaptcha.subscribe((res) => {
      if (res != null) {
        this.isActiveCaptcha = res;
      }
    });

    this._authSettingStore.isActiveCaptchaBehavior.subscribe((res) => {
      if (res != null) {
        this.isActiveCaptcha = res;
      }
    });
  }

  ngOnDestroy(): void {
    this.subscriptions.unsubscribe();
  }

  submit() {


    if (
      (this.captcha &&
        this.captcha.captchaInfo &&
        this.captcha.captchaInfo.captchaId &&
        this.captcha.captchaInfo.captchaCode && this.model.password && this.model.username)
      || (!this.isActiveCaptcha && this.model.password && this.model.username)
    ) {
      this.model.username = toEnglishDigits(this.model.username);
      if (!isValidEmail(this.model.username) && !isValidMobile(this.model.username)) {
        this._alert.error('ایمیل وارد شده صحیح نمی باشد');
        return;
      }
      if (this.isActiveCaptcha) {
        const { captchaCode, captchaId } = this.captcha.captchaInfo;
        this.model.captchaCode = captchaCode;
        this.model.captchaCode = toEnglishDigits(this.model.captchaCode)
        this.model.captchaId = captchaId;
      } else {
        this.model.captchaCode = null
        this.model.captchaId = null
      }
      this.isLoading = true;
      this.subscriptions.add(
        this._commonService
          .putforlogin(Address.login, this.model)
          .pipe(
            finalize(() => {
              this.isLoading = false;
              this.captcha.handleRefreshCaptcha();
            })
          )
          .subscribe(
            (response: response) => {
              setStorageItemCallback(
                ColumnKey.UserToken,
                {
                  access_token: response.data.access_token,
                  refresh_token: response.data.refresh_token,
                },
                () => {
                  this._redirectorHelper.routerHandlerAfterAuth(
                    response.data.access_token
                  );
                }
              );
            },
            (error) => {
              if (error.status == 481 || error.status == 482) {
                this._usernameStore.username.next(this.model.username);
                this._usernameStore.usernameBehavior.next(this.model.username);
                this._router.navigateByUrl('/login');
              }
            }
          )
      );
    }
  }


}
