import { Component, ViewChild } from '@angular/core';
import { ActivatedRoute, Router, RoutesRecognized } from '@angular/router';
import { Address } from '@enums/address.enum';
import { RegisterModel } from '@models/auth/register';
import { CommonService } from '@services/common.service';
import { Subscription } from 'rxjs';
import { filter, finalize, pairwise, tap } from 'rxjs/operators';
import { response } from '@interfaces/response';
import { CaptchaComponent } from '@components/captcha/captcha.component';
import { setSessionStorageItem } from '@helpers/sessionStorage.helper';
import { ColumnKey } from '@enums/columnKey.enum';
import { isInteger, isValidEmail } from '@helpers/validator.helper';
import { AlertHelper } from '@helpers/alert.helper';
import { RedirectorHelper } from '@helpers/redirector.helper';
import { AuthStore } from '@stores/AuthSetting.store';
@Component({
  selector: 'form-register',
  templateUrl: './formRegister.component.html',
  styleUrls: ['./formRegister.component.scss'],
})
export class FormRegisterComponent {
  isLoading: boolean = false;
  subscriptions = new Subscription();
  isNumber: boolean = true;
  model: RegisterModel = new RegisterModel();

  isActiveCaptcha: boolean;
  @ViewChild('captcha') captcha: CaptchaComponent;

  constructor(
    private _commonService: CommonService,

    private _router: Router,
    private _alert: AlertHelper,
    public _redirectorHelper: RedirectorHelper,
    private _authSettingStore: AuthStore
  ) {
    this.subscriptions = this._router.events
      .pipe(
        filter((evt: any) => evt instanceof RoutesRecognized),
        tap((event) => { }),
        pairwise()
      )
      .subscribe((events: RoutesRecognized[]) => { });

    this._authSettingStore.isActiveCaptcha.subscribe((res) => {
      if (res != null) {
        this.isActiveCaptcha = res;
      }
    });

    this._authSettingStore.isActiveCaptchaBehavior.subscribe((res) => {
      if (res != null) {
        this.isActiveCaptcha = res;
      }
    });
  }
  getSubmitMethod(): any {
    return {
      callSubmitMethod: () => {
        this.submit();
      },
    };
  }
  ngOnDestroy(): void {
    this.subscriptions.unsubscribe();
  }

  submit() {
    if (isInteger(this.model.address)) this.model.typeVerify = 101;
    else {
      if (!isValidEmail(this.model.address)) {
        this._alert.warning('ایمیل وارد شده صحیح نمی باشد');
        return;
      }
      this.model.typeVerify = 102;
    }

    if (this.isActiveCaptcha) {
      const { captchaCode, captchaId } = this.captcha.captchaInfo;
      this.model.captchaCode = captchaCode;
      this.model.captchaId = captchaId;
    } else {
      this.model.captchaCode = null
      this.model.captchaId = null
    }
    this.isLoading = true;

    this.subscriptions.add(
      this._commonService
        .post(Address.Send, this.model, false)
        .pipe(
          finalize(() => {
            this.isLoading = false;
            this.captcha.handleRefreshCaptcha();
          })
        )
        .subscribe(
          (response: response) => {
            setSessionStorageItem(ColumnKey.Address, response.data.address);
            setSessionStorageItem(ColumnKey.TokenVerify, response.data.token);
            setSessionStorageItem(
              ColumnKey.TypeVerify,
              response.data.typeVerify
            );
          },
          (error) => { }
        )
    );
  }
}
