import { Component, OnDestroy, OnInit, ViewChild } from '@angular/core';
import { CommonService } from '@services/common.service';
import { Subscription } from 'rxjs';
import { finalize, filter, pairwise, tap } from 'rxjs/operators';
import { response } from '@interfaces/response';
import { Router, RoutesRecognized } from '@angular/router';
import { Address } from '@enums/address.enum';
import {
  getSessionStorageItem,
  setSessionStorageItem,
} from '@helpers/sessionStorage.helper';
import { ColumnKey } from '@enums/columnKey.enum';
import {
  CountdownComponent,
  CountdownConfig,
  CountdownEvent,
} from 'ngx-countdown';
import { AlertHelper } from '@helpers/alert.helper';
import { VerifyModel } from 'src/app/core/params/auth/verifyMode';
import { setStorageItem } from '@helpers/localstorage.helper';
import { RedirectorHelper } from '@helpers/redirector.helper';
import { stringIsEmptyOrNull } from '@helpers/validator.helper';
@Component({
  selector: 'verify',
  templateUrl: './verify.component.html',
  styleUrls: ['./verify.component.scss'],
})
export class VerifyComponent implements OnInit, OnDestroy {
  isLoading: boolean = false;
  isLoadingResendCode: boolean = false;
  canResend: boolean = false;
  leftTime: number;
  otp: string;

  setInterval = setInterval;
  subscriptions = new Subscription();
  model: VerifyModel = new VerifyModel();
  @ViewChild('cd', { static: false }) private countdown: CountdownComponent;

  config: CountdownConfig = {
    leftTime: 0,
    format: 'm:s',
  };
  setNewTimer(type: number) {
    switch (type) {
      case 101:
        this.leftTime = 120;
        break;
      case 102:
        this.leftTime = 240;
        break;
    }
    this.config = {
      leftTime: this.leftTime,
      format: 'm:s',
    };
    this.canResend = false;
  }

  onOtpChange(otp) {
    this.model.code = otp;
    if (this.model.code.length == 5) {
      this.submit();
    }
  }

  finishTimer(e: CountdownEvent) {
    if (e.action == 'done') {
      this.canResend = true;
    }
  }
  constructor(
    private _commonService: CommonService,
    private _router: Router,
    private _alert: AlertHelper,
    private _redirectorHelper: RedirectorHelper
  ) {
    this.subscriptions = this._router.events
      .pipe(
        filter((evt: any) => evt instanceof RoutesRecognized),
        tap((event) => {}),
        pairwise()
      )
      .subscribe((events: RoutesRecognized[]) => {});
  }

  ngOnInit(): void {
    this.model.token = getSessionStorageItem(ColumnKey.TokenVerify);
    this.model.typeVerify = Number(getSessionStorageItem(ColumnKey.TypeVerify));
    this.setNewTimer(this.model.typeVerify);
    this.model.address = getSessionStorageItem(ColumnKey.Address);
  }
  ngOnDestroy(): void {
    this.subscriptions.unsubscribe();
  }

  onCaptchaResolved(response: string): void {
    console.log('reCAPTCHA v2 Response:', response);
  }
  onCaptchaErrored(e: any) {
    console.warn(`reCAPTCHA error encountered`);
  }

  reSendCode() {
    if (stringIsEmptyOrNull(this.model.tokenRecaptcha)) {
      this._alert.error('لطفا کپچا را تایید کنید');
      return;
    }
    this.isLoadingResendCode = true;
    this.subscriptions.add(
      this._commonService
        .post(Address.ResendCode, this.model, false)
        .pipe(
          finalize(() => {
            this.isLoadingResendCode = false;
          })
        )
        .subscribe(
          (response: response) => {
            this._alert.success('کد یکبار مصرف مجددا ارسال شد');
            setTimeout(() => {
              this.setNewTimer(this.model.typeVerify);
            }, 1000);
            this.model.token = response.data.token;
            setSessionStorageItem(ColumnKey.TokenVerify, response.data.token);
            setSessionStorageItem(ColumnKey.Address, response.data.address);
            setSessionStorageItem(
              ColumnKey.TypeVerify,
              response.data.typeVerify
            );
          },
          (error) => {
            this._router.navigate(['/verify']);
          }
        )
    );
  }

  submit() {
    this.isLoading = true;
    this.subscriptions.add(
      this._commonService
        .post(Address.Verify, this.model, false)
        .pipe(
          finalize(() => {
            this.isLoading = false;
          })
        )
        .subscribe(
          (response: response) => {
            setStorageItem(ColumnKey.UserToken, {
              access_token: response.data.access_token,
              refresh_token: response.data.refresh_token,
            });
            // this._redirectorHelper.routerHandlerAfterAuth();
          },
          (error) => {}
        )
    );
  }
  back() {
    window.history.back();
  }
}
