import { stringIsEmptyOrNull } from './../../core/helpers/validator.helper';
import {
  Component,
  EventEmitter,
  NgModule,
  OnInit,
  Output,
  Input,
  OnChanges,
  ChangeDetectorRef,
  ChangeDetectionStrategy,
} from '@angular/core';
import { CommonService } from '@services/common.service';
import { Address } from '@enums/address.enum';
import { finalize } from 'rxjs/operators';
import { Subscription } from 'rxjs';
import { ReactiveFormsModule, FormsModule } from '@angular/forms';
import { CommonModule } from '@angular/common';
import { TooltipModule } from 'primeng/tooltip';
import { NgxMaskModule } from 'ngx-mask';
import { SkeletonModule } from 'primeng/skeleton';
import { InputTextModule } from 'primeng/inputtext';
import {
  convertToYearJalali,
  convertNumberDayOfMonth,
  convertDateToGeorgian,
  convertDateToJalali,
} from '@helpers/dateTime.helper';
import { DropdownModule } from 'primeng/dropdown';
import { getMonth } from '@helpers/dataStatic.helper';
import { AlertHelper } from '@helpers/alert.helper';
import { AutoUnsubscribe } from 'src/app/core/decorators/AutoUnSubscribe';
import { UserStore } from '@stores/user.store';
import { UserProfile } from '@interfaces/user/userProfile';

@AutoUnsubscribe({ arrayName: 'subscriptions' })
@Component({
  selector: 'date-picker-select',
  templateUrl: './datePickerSelect.component.html',
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class DatePickerSelectComponent implements OnInit, OnChanges {
  isLoading: boolean = true;
  subscriptions: Subscription[] = [];
  first: boolean = true;
  year: number = -1;
  month: number = -1;
  day: string = '-1';
  dayList: any[] = [];
  yearList: any[] = [];
  monthList: any[] = getMonth();
  currentYear: number;
  @Input() isRequired: boolean = false;
  @Input() submitted: boolean = false;
  @Input() minusYear: number = 100;
  @Input() plusYear: number = 50;
  @Input() dateValue: string;
  @Input() title: string = null;
  @Output() dateValueChange: EventEmitter<string> = new EventEmitter<string>();
  userInfo: UserProfile = new UserProfile();
  constructor(
    private _commonService: CommonService,
    private _alert: AlertHelper,
    private _userStore: UserStore,
    private cdr: ChangeDetectorRef
  ) {
    this._userStore.user.subscribe((res) => {
      if (res != null) {
        this.userInfo = res;
      }
    });
    this._userStore.getUserBehavior().subscribe((res) => {
      if (res != null) {
        this.userInfo = res;
      }
    });
  }
  ngOnChanges() {
    if (!stringIsEmptyOrNull(this.dateValue)) {
      this.dateValueChange.emit(this.dateValue);
      let dateJalali = convertDateToJalali(this.dateValue);
      this.year = Number(dateJalali.split('/')[0]);
      this.month = Number(dateJalali.split('/')[1]);
      if (
        Number(dateJalali.split('/')[2]) < 10 &&
        dateJalali.split('/')[2].length > 1
      ) {
        this.day = String(dateJalali.split('/')[2].substring(1, 2));
      } else this.day = String(dateJalali.split('/')[2]);
      this.cdr.detectChanges();
    }
  }
  ngOnInit() {
    for (let i = 1; i <= 31; i++) {
      this.dayList.push({ name: String(i) });
    }
    this.getDate();
  }

  selectDate() {
    if (
      this.month != null &&
      this.year != null &&
      !isNaN(this.month) &&
      Number(this.month) != -1 &&
      !isNaN(this.year) &&
      this.year != -1
    ) {
      let dayNumber = convertNumberDayOfMonth(this.year, this.month);
      if (this.day != null && Number(dayNumber) < Number(this.day)) {
        this._alert.warning('تاریخ انتخابی صحیح نمی باشد');
        this.year = null;
        this.month = null;
      }
      if (this.dayList == null || this.dayList.length == 0) {
        for (let i = 1; i <= Number(dayNumber); i++) {
          this.dayList.push({ name: String(i) });
        }
      }
      if (this.dayList.length < 30 && Number(dayNumber) > 29) {
        this.dayList.push({ name: String(30) });
      }
      if (this.dayList.length < 31 && Number(dayNumber) > 30) {
        this.dayList.push({ name: String(31) });
      }
    }

    if (
      this.month != null &&
      this.year != null &&
      this.day != null &&
      Number(this.month) != -1 &&
      this.year != -1 &&
      this.day != '-1'
    ) {
      this.dateValueChange.emit(
        convertDateToGeorgian(this.year + '/' + this.month + '/' + this.day)
      );
    }
    this.cdr.detectChanges();
  }

  getDate() {
    this.isLoading = true;
    this.subscriptions.push(
      this._commonService
        .get(Address.DateTime)
        .pipe(
          finalize(() => {
            this.isLoading = false;
          })
        )
        .subscribe((response) => {
          let currentYear = Number(convertToYearJalali(response));
          this.currentYear = currentYear;
          let counter = 0;
          if (this.plusYear != null && this.plusYear > 0) {
            counter = currentYear + this.plusYear;
            while (counter > currentYear + 1) {
              counter--;
              this.yearList.push({ name: String(counter), id: counter });
            }
          }
          if (this.minusYear != null && this.minusYear > 0) {
            let minYear = currentYear - this.minusYear;
            counter = currentYear;
            while (counter > minYear) {
              this.yearList.push({ name: String(counter), id: counter });
              counter--;
            }
            this.yearList.push({ name: String(currentYear - this.minusYear) });
          } else this.yearList.push({ name: String(currentYear), id: counter });
          this.cdr.detectChanges();
        })
    );
  }
}

@NgModule({
  declarations: [DatePickerSelectComponent],
  imports: [
    ReactiveFormsModule,
    FormsModule,
    SkeletonModule,
    CommonModule,
    TooltipModule,
    NgxMaskModule,
    SkeletonModule,
    InputTextModule,
    DropdownModule,
  ],
  exports: [DatePickerSelectComponent],
})
export class DatePickerSelectModule {}
