import { Injectable, OnDestroy } from '@angular/core';
import { Observable, Subscription, throwError } from 'rxjs';
import { catchError } from 'rxjs/operators';
import { HttpErrorResponse, HttpClient } from '@angular/common/http';
import { Address } from '@enums/address.enum';
import { isGuid } from '@helpers/validator.helper';
import { CommonService } from './common.service';
import { FileUrlGeneratorStore } from '@stores/fileUrlGenerator.store';

@Injectable({ providedIn: 'root' })
export class FileUploadService implements OnDestroy {
  private subscriptions = new Subscription();

  constructor(
    private http: HttpClient,
    private _commonService: CommonService,
    private _fileUrlGeneratorStore: FileUrlGeneratorStore
  ) {}

  ngOnDestroy() {
    this.subscriptions.unsubscribe();
  }

  fileUpload(name: string, file: File, fileUploadUrl: string): Observable<any> {
    const formData = new FormData();
    formData.append('name', name);
    formData.append('file', file);

    return this.http
      .post(fileUploadUrl, formData, {
        reportProgress: true,
        observe: 'events',
      })
      .pipe(catchError(this.errorMgmt));
  }

  private setFileUrlsDetection(
  list: any[],
  fileName: string,
  storeNext: (list: any) => void,
  storeBehavior: (list: any) => void,
  observer?: any
) {
  if (list == undefined || list == null) {
    if (observer) observer.complete();
    return;
  }
  storeNext(null);
  storeBehavior(null);

  const listIds = list
    .filter((item) => isGuid(item[fileName]))
    .map((item) => item[fileName]);

  if (!listIds.length) {
    if (observer) observer.complete();
    return;
  }

  this.subscriptions.add(
    this._commonService
      .put(Address.GetFileUrlGetGroup, { ids: listIds })
      .subscribe({
        next: (response) => {
          for (let item of list) {
            const match = response.find(
              (res: any) => res.fileId === item[fileName]
            );
            if (match) item.fileUrl = match.url;
          }
          storeNext(list);
          storeBehavior(list);

          if (observer) {
            observer.next(list);
            observer.complete();
          }
        },
        error: (error) => {
          if (observer) observer.error(error);
        }
      })
  );
}

  private setFileUrls(
    list: any[],
    fileName: string,
    storeNext: (list: any) => void,
    storeBehavior: (list: any) => void
  ) {
    if (list == undefined || list == null) {
      return;
    }
    storeNext(null);
    storeBehavior(null);

    const listIds = list
      .filter((item) => isGuid(item[fileName]))
      .map((item) => item[fileName]);

    if (!listIds.length) return;

    this.subscriptions.add(
      this._commonService
        .put(Address.GetFileUrlGetGroup, { ids: listIds })
        .subscribe((response) => {
          for (let item of list) {
            const match = response.find(
              (res: any) => res.fileId === item[fileName]
            );
            if (match) item.fileUrl = match.url;
          }
          storeNext(list);
          storeBehavior(list);
        })
    );
  }

  fileUrlGeneratorPutBankCard(list: any[], fileName: string) {
    this.setFileUrls(
      list,
      fileName,
      this._fileUrlGeneratorStore.listWithFileUrlCardBank.next.bind(
        this._fileUrlGeneratorStore.listWithFileUrlCardBank
      ),
      this._fileUrlGeneratorStore.listWithFileUrlCardBankBehavior.next.bind(
        this._fileUrlGeneratorStore.listWithFileUrlCardBankBehavior
      )
    );
  }

  fileUrlGeneratorPutCountryDetection(list: any[], fileName: string): Observable<any> {
  return new Observable(observer => {
    this.setFileUrlsDetection(
      list,
      fileName,
      this._fileUrlGeneratorStore.listWithFileUrlCountry.next.bind(
        this._fileUrlGeneratorStore.listWithFileUrlCountry
      ),
      this._fileUrlGeneratorStore.listWithFileUrlCountryBehavior.next.bind(
        this._fileUrlGeneratorStore.listWithFileUrlCountryBehavior
      ),
      observer
    );
  });
}

  fileUrlGeneratorPutCountry(list: any[], fileName: string) {
    this.setFileUrls(
      list,
      fileName,
      this._fileUrlGeneratorStore.listWithFileUrlCountry.next.bind(
        this._fileUrlGeneratorStore.listWithFileUrlCountry
      ),
      this._fileUrlGeneratorStore.listWithFileUrlCountryBehavior.next.bind(
        this._fileUrlGeneratorStore.listWithFileUrlCountryBehavior
      )
    );
  }

  fileUrlGeneratorPutBankAvatar(list: any[], fileName: string) {
    this.setFileUrls(
      list,
      fileName,
      this._fileUrlGeneratorStore.listWithFileUrlAdvertising.next.bind(
        this._fileUrlGeneratorStore.listWithFileUrlAdvertising
      ),
      this._fileUrlGeneratorStore.listWithFileUrlAdvertisingBehavior.next.bind(
        this._fileUrlGeneratorStore.listWithFileUrlAdvertisingBehavior
      )
    );
  }

  fileUrlGeneratorPutTicket(list: any[], fileName: string) {
    this.setFileUrls(
      list,
      fileName,
      this._fileUrlGeneratorStore.listWithFileUrlTicket.next.bind(
        this._fileUrlGeneratorStore.listWithFileUrlTicket
      ),
      this._fileUrlGeneratorStore.listWithFileUrlTicketBehavior.next.bind(
        this._fileUrlGeneratorStore.listWithFileUrlTicketBehavior
      )
    );
  }

  fileUrlGeneratorPutCurrencyCrypto(list: any[], fileName: string) {
    this.setFileUrls(
      list,
      fileName,
      this._fileUrlGeneratorStore.listWithFileUrlCurrencyCrypto.next.bind(
        this._fileUrlGeneratorStore.listWithFileUrlCurrencyCrypto
      ),
      this._fileUrlGeneratorStore.listWithFileUrlCurrencyCryptoBehavior.next.bind(
        this._fileUrlGeneratorStore.listWithFileUrlCurrencyCryptoBehavior
      )
    );
  }

  fileUrlGeneratorPutOrderDocument(list: any[], fileName: string) {
    this.setFileUrls(
      list,
      fileName,
      this._fileUrlGeneratorStore.listWithFileUrlOrderDocument.next.bind(
        this._fileUrlGeneratorStore.listWithFileUrlOrderDocument
      ),
      this._fileUrlGeneratorStore.listWithFileUrlOrderDocumentBehavior.next.bind(
        this._fileUrlGeneratorStore.listWithFileUrlOrderDocumentBehavior
      )
    );
  }

  fileUrlGeneratorPutAdvertising(list: any[], fileName: string) {
    this.setFileUrls(
      list,
      fileName,
      this._fileUrlGeneratorStore.listWithFileUrlAdvertising.next.bind(
        this._fileUrlGeneratorStore.listWithFileUrlAdvertising
      ),
      this._fileUrlGeneratorStore.listWithFileUrlAdvertisingBehavior.next.bind(
        this._fileUrlGeneratorStore.listWithFileUrlAdvertisingBehavior
      )
    );
  }

  private errorMgmt(error: HttpErrorResponse) {
    return throwError(() =>
      error.error instanceof ErrorEvent
        ? error.error.message
        : `Error Code: ${error.status}\nMessage: ${error.message}`
    );
  }
}
