import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  EventEmitter,
  Input,
  OnDestroy,
  OnInit,
  Output,
} from '@angular/core';

import { DeviceDetectorService } from 'ngx-device-detector';

import { CountdownConfig, CountdownEvent } from 'ngx-countdown';
import { timeDiffrential } from '@helpers/dateTime.helper';
import { Router } from '@angular/router';
import { BlockStore } from '@stores/blockOrder';
import { Subscription, finalize } from 'rxjs';
import { Address } from '@enums/address.enum';
import { CommonService } from '@services/common.service';

@Component({
  selector: 'countdown-order',
  templateUrl: 'countdownOrder.component.html',
  styleUrls: ['countdownOrder.component.scss'],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class CountdownOrderComponent implements OnInit, OnDestroy {
  constructor(
    public _deviceDetectorService: DeviceDetectorService,
    private cdr: ChangeDetectorRef,

    private _blockStore: BlockStore,
    private _commonService: CommonService
  ) {}
  subscriptions = new Subscription();
  @Input() deadlineSencond: any;
  @Input() timeNow: any;
  @Input() expired: boolean;
  @Input() orderId: any;
  isCanceled: boolean = false;
  @Output() expiredChange: EventEmitter<boolean> = new EventEmitter<boolean>();

  config: CountdownConfig = {
    leftTime: 0,
    format: 'm:s',
  };

  ngOnDestroy() {
    this.subscriptions.unsubscribe();
  }

  setNewTimer() {
    let totalSeconds = timeDiffrential(this.timeNow, this.deadlineSencond);

    if (totalSeconds > 0) {
      this.config = {
        leftTime: totalSeconds,
        format: 'm:s',
      };
      this.expiredChange.emit(false);
    } else {
      this.expiredChange.emit(true);
    }

    this.cdr.detectChanges();
  }

  finishTimer(e: CountdownEvent) {
    if ((e.action == 'done' || !e.left) && !this.isCanceled) {
      this.isCanceled = true;
      this.expiredChange.emit(true);

      this._blockStore.expireOrder.next(true);
      this._blockStore.expireOrderBehavior.next(true);
      this.cancel(this.orderId);
    }
    this.cdr.detectChanges();
  }

  ngOnInit() {
    this.setNewTimer();
  }

  cancel(orderId: any) {
    this.subscriptions.add(
      this._commonService
        .put(Address.OrderCancelExpiredDate, { orderId: orderId })
        .pipe(finalize(() => {}))
        .subscribe(
          (res) => {},
          (error: any) => {},
          () => {}
        )
    );
  }
}
