import { isGuid, stringIsEmptyOrNull } from '@helpers/validator.helper';
import { Static } from '@interfaces/static';
import { Router, ActivatedRoute } from '@angular/router';
import {
  Component,
  OnDestroy,
  OnInit,
  ChangeDetectorRef,
  OnChanges,
} from '@angular/core';
import { Address } from '@enums/address.enum';
import { AlertHelper } from '@helpers/alert.helper';
import { CommonService } from '@services/common.service';
import { finalize, Subscription } from 'rxjs';
import { OrderModel } from '@models/orders/OrderModel';
import { response } from '@interfaces/response';
import { HttpEvent, HttpEventType } from '@angular/common/http';
import { AccountNumberModel } from '@models/orders/AccountNumberModel';
import { getDepositOptions } from '@helpers/values.helper';
import { OrderDocumentModel } from 'src/app/core/params/order/OrderDocumentModel';
import { FileUploadService } from '@services/fileUpload.service';
import { BlockStore } from '@stores/blockOrder';
import { cloneDeep } from 'lodash-es';
import { GetScenarioCheckout } from '@models/orders/getScenarioCheckoutModel';
import { TranslateService } from '@ngx-translate/core';

@Component({
  selector: 'app-deposit-document',
  templateUrl: './depositDocument.component.html',
  styleUrls: ['./depositDocument.component.scss'],
})
export class DepositDocumentComponent implements OnInit, OnDestroy, OnChanges {
  subscriptions = new Subscription();

  constructor(
    private _alert: AlertHelper,
    private _commonService: CommonService,
    private cdr: ChangeDetectorRef,
    private _router: Router,
    private _activatedroute: ActivatedRoute,
    private _fileUploadService: FileUploadService,
    public _blockStore: BlockStore,
    private _translateService: TranslateService
  ) {}

  isRedirectig: boolean;
  documentType: number = 1;
  order: OrderModel;
  accountNumber: AccountNumberModel = new AccountNumberModel();
  orderDocument: OrderDocumentModel = new OrderDocumentModel();
  isLoadingGetOrder: boolean = false;
  isLoadingGetAccountNumber: boolean = false;
  isLoading: boolean = false;
  isLoadingFile: boolean = false;
  isLoadingGetFile: boolean = false;
  extensionStore: string;
  warnText: string;
  canNotUpload: boolean = false;
  url: string = null;
  file: File;
  progress: number = 0;
  payOnline: boolean;
  expireDate: any;
  dateNow: any;
  isExpired: boolean;
  dataScenario: GetScenarioCheckout = new GetScenarioCheckout();
  public scenarioFinalAmount: number;
  stateDepositDocument: Static[] = getDepositOptions();

  ngOnInit(): void {
    if (!isGuid(this._activatedroute.snapshot.params['id']))
      this._router.navigateByUrl('/dashboard/orders');
    this.orderDocument.orderId = this._activatedroute.snapshot.params['id'];
    this.getOrder();
  }

  updateStatusWaiting() {
    this.subscriptions.add(
      this._commonService
        .put(
          Address.OrderUpdateWaitingStatus,
          { orderId: this._activatedroute.snapshot.params['id'] },
          false
        )
        .pipe(
          finalize(() => {
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            if (response) {
              setTimeout(() => {
                this._router.navigateByUrl('/dashboard/orders');
              }, 4000);
            }
          },
          (error) => {}
        )
    );
  }
  storeBlockOrder() {
    if (
      (this.order.orderType == '907' ||
        this.order.orderType == '171' ||
        this.order.orderType == '940' ||
        this.order.orderType == '167') &&
      Number(this.order.status) != 193
    ) {
      this._blockStore.blockOrder.next({
        statusName: this.order.statusName,
        status: this.order.status,
        currentStatusPage: 1,
      });

      this._blockStore.blockOrderBehavior.next({
        statusName: this.order.statusName,
        status: this.order.status,
        currentStatusPage: 1,
      });
    } else if (
      Number(this.order.status) != 188 &&
      this.order.orderType != '907' &&
      this.order.orderType != '171' &&
      this.order.orderType != '940' &&
      this.order.orderType != '167'
    ) {
      this._blockStore.blockOrderBehavior.next({
        statusName: this.order.statusName,
        status: this.order.status,
        currentStatusPage: 1,
      });
    }
    this.cdr.detectChanges();
  }

  ngOnDestroy() {
    this.subscriptions.unsubscribe();
    this._blockStore.expireOrderBehavior.next({});
    this._blockStore.expireOrder.next({});
    this._blockStore.blockOrderBehavior.next({});
    this._blockStore.blockOrder.next({});
    this.cdr.detectChanges();
  }

  ngOnChanges() {
    if (this.isExpired == false || this.isExpired == true) {
      this._blockStore.expireOrderBehavior.next(this.isExpired);
      this._blockStore.expireOrder.next(this.isExpired);
    }
    this.cdr.detectChanges();
  }

  redirectToPaymentGateway() {
    this.isRedirectig = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(
          Address.changeStatusPayOrder,
          { orderId: this._activatedroute.snapshot.params['id'] },
          false
        )
        .pipe(
          finalize(() => {
            this.isRedirectig = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            if (response.data && response.data.redirectUrl) {
              window.location.href = response.data.redirectUrl;
            }

            this.cdr.detectChanges();
          },
          (error) => {}
        )
    );
  }

  fileChange(event) {
    let fileList: FileList =
      event.target != undefined && event.target != null
        ? event.target.files
        : event;
    if (fileList.length > 0) {
      this.file = fileList[0];
      this.fileUpload();
      this.cdr.detectChanges();
    }
  }

  fileUpload() {
    this.isLoadingFile = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._fileUploadService
        .fileUpload(this.file.name, this.file, Address.FileUploadDocument)
        .subscribe((event: HttpEvent<any>) => {
          switch (event.type) {
            case HttpEventType.Sent:
              break;
            case HttpEventType.ResponseHeader:
              break;
            case HttpEventType.UploadProgress:
              let eventTotal = event.total ? event.total : 0;
              this.progress = Math.round((event.loaded / eventTotal) * 100);
              this.cdr.detectChanges();
              break;
            case HttpEventType.Response:
              if (event.body['data'] != null) {
                this.orderDocument.imageId = event.body['data'];
                this.loadFiles(event.body['data']);
                this.isLoadingFile = false;
                this.cdr.detectChanges();
              }
              setTimeout(() => {
                this.progress = 0;
                this.cdr.detectChanges();
              }, 1500);
          }
        })
    );
  }

  loadFiles(id: string) {
    this.isLoadingGetFile = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .getResponseText(Address.GetFileUrl + id)
        .pipe(
          finalize(() => {
            this.isLoadingFile = false;
            this.isLoadingGetFile = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response) => {
            this.url = response;
            this.extensionStore = String(
              this.url.substring(this.url.lastIndexOf('.') + 1, this.url.length)
            ).toLowerCase();
            this.cdr.detectChanges();
          },
          (err: any) => {},
          () => {}
        )
    );
  }

  remove() {
    this.url = null;
  }

  getOrder() {
    this.isLoadingGetOrder = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(
          Address.OrderGetbyId,
          { orderId: this._activatedroute.snapshot.params['id'] },
          false
        )
        .pipe(
          finalize(() => {
            this.isLoadingGetOrder = false;
            this.getAccountNUmber();
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this.order = response.data;
            this.cdr.detectChanges();
          },
          (error) => {}
        )
    );
  }

  detectChanges() {
    this.cdr.detectChanges();
  }

  getAccountNUmber() {
    this.isLoadingGetAccountNumber = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(
          Address.AccountNumberGet,
          {
            orderId: this._activatedroute.snapshot.params['id'],
            isSmartPayment: true,
            code: this.order.discountCode,
          },
          false
        )
        .pipe(
          finalize(() => {
            this.isLoadingGetAccountNumber = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this._fileUploadService.fileUrlGeneratorPutCountry(
              [response.data],
              'bankAvatarId'
            );
            this.accountNumber = response.data;
            if (
              this.order &&
              this.accountNumber &&
              this.accountNumber.description
            ) {
              this.accountNumber.description =
                this.accountNumber.description.replace(
                  '@number',
                  this.order.orderNumber
                );
            }

            if (
              this.accountNumber &&
              !this.accountNumber.cardNumber &&
              !this.accountNumber.depositNumber &&
              !this.accountNumber.shebaNumber &&
              ![167, 171, 959, 998, 969, 940, 906, 907].includes(
                Number(this.order.orderType)
              )
            ) {
              this.updateStatusWaiting();
            }

            if (this.order) {
              this.orderScenarioGetAll();
              this.storeBlockOrder();
            }

            this.cdr.detectChanges();
          },
          (error) => {}
        )
    );
  }

  submit() {
    if (
      this.accountNumber &&
      !this.accountNumber.isSmartPayment &&
      this.orderDocument.imageId == null &&
      stringIsEmptyOrNull(this.orderDocument.trackingCode)
    ) {
      this._alert.warning(
        this._translateService.translations[this._translateService.currentLang][
          'enterPaymentDoc'
        ]
      );
      return;
    }

    this.isLoading = true;
    this.cdr.detectChanges();

    this.subscriptions.add(
      this._commonService
        .post(Address.OrderDocument, this.orderDocument)
        .pipe(
          finalize(() => {
            this.isLoading = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this._alert.success(
              this._translateService.translations[
                this._translateService.currentLang
              ]['enteredDocWillBeChecked']
            );
            this.canNotUpload = true;

            const companyId = this._activatedroute.snapshot.params['company'];

            this._router.navigateByUrl(
              '/dashboard/orders/invoices/' + companyId
            );
          },
          (error) => {}
        )
    );
  }

  documentTypeChange(e) {
    switch (e.value) {
      case 1:
        this.orderDocument.trackingCode = null;
        break;
      case 2:
        this.orderDocument.imageId = null;
        this.url = null;
        break;
      default:
        break;
    }
    this.cdr.detectChanges();
  }

  recieveFinishCountDown($event: boolean) {
    this.isExpired = $event;
  }

  checkForScenario() {
    if (this.dataScenario.isAllowed) {
      if (this.order.discount) {
        if (this.dataScenario.paymentOfOrderWage) {
          this.scenarioFinalAmount = this.order.finalWageAmount;
        } else if (this.dataScenario.paymentWithoutOrderWage) {
          this.scenarioFinalAmount =
            this.order.discountedFinalAmount - this.order.finalWageAmount;
        } else {
          this.scenarioFinalAmount = this.order.finalAmount;
        }
      } else if (!this.order.discount) {
        if (this.dataScenario.paymentOfOrderWage) {
          this.scenarioFinalAmount = this.order.finalWageAmount;
        } else if (this.dataScenario.paymentWithoutOrderWage) {
          this.scenarioFinalAmount =
            this.order.finalAmount - this.order.finalWageAmount;
        } else {
          this.scenarioFinalAmount = this.order.finalAmount;
        }
      }
    } else {
      this.scenarioFinalAmount = this.order.finalAmount;
    }
    this.cdr.detectChanges();
  }

  orderScenarioGetAll() {
    this.subscriptions.add(
      this._commonService
        .put(Address.ScenarioGetAll, {
          orderType: this.order.orderType,
        })
        .pipe(finalize(() => {}))
        .subscribe((response) => {
          if (response.data) {
            const dataScenario = response.data;
            if (this.order) {
              const finalAmount = this.order.finalAmount;
              const paidAmount = this.order.paidAmount;
              const isPaid = paidAmount && paidAmount !== 0;
              const targetSortNumber = isPaid ? 2 : 1;
              const validScenarios = dataScenario.filter((item: any) => {
                const isSortMatch = item.sortNumber === targetSortNumber;
                const minValid =
                  !item.minimumAmountPaidToStart ||
                  finalAmount >= +item.minimumAmountPaidToStart;

                const maxValid =
                  !item.maximumAmountPaidToFinish ||
                  finalAmount < +item.maximumAmountPaidToFinish;
                return isSortMatch && minValid && maxValid;
              });
              const selectedScenario = cloneDeep(validScenarios[0]);
              if (selectedScenario) {
                this.dataScenario = selectedScenario;
                this.dataScenario.isAllowed = true;
              }
              this.checkForScenario();
              this.cdr.detectChanges();
            }
          }
        })
    );
  }
}
