import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  Input,
  OnDestroy,
  OnInit,
} from '@angular/core';
import { ActivatedRoute } from '@angular/router';
import { Address } from '@enums/address.enum';
import { MessageStatic } from '@enums/messageStatic.enum';
import { CommonService } from '@services/common.service';
import { Location } from '@angular/common';
import { ConfirmationService, Message } from 'primeng/api';
import { DialogService } from 'primeng/dynamicdialog';
import { Subscription, finalize } from 'rxjs';
import { OrderSearchModel } from 'src/app/core/params/order/OrderSearchModel';
import { BaseParams } from 'src/app/core/params/user/baseParams';
import { fadeAnimation } from 'src/app/core/animations/animations';
import { WalletModelGet } from '@models/user/WalletModelGet';
import { TableLazyLoadEvent } from 'primeng/table';
import { PagingHelper } from '@helpers/paging.helper';
import { UserProfile } from '@interfaces/user/userProfile';
import { response } from '@interfaces/response';
import { AlertHelper } from '@helpers/alert.helper';
import { AddUserModalComponent } from './addUserModal/addUserModal.component';
import { Icon } from '@enums/iconKey.enum';
import { stringIsEmptyOrNull } from '@helpers/validator.helper';
import { ColumnKey } from '@enums/columnKey.enum';
import { cloneDeep } from 'lodash-es';

@Component({
  selector: 'add-user',
  templateUrl: './addUser.component.html',
  styleUrls: ['./addUser.component.scss'],
  changeDetection: ChangeDetectionStrategy.OnPush,
  animations: [fadeAnimation],
  host: { '[@fadeInAnimation]': '' },
})
export class AddUserComponent implements OnInit, OnDestroy {
  constructor(
    private _commonService: CommonService,
    private cdr: ChangeDetectorRef,
    public dialogService: DialogService,
    public _activatedroute: ActivatedRoute,
    public _pagingHelper: PagingHelper,
    public location: Location,
    private _messageService: AlertHelper,
    private confirmationService: ConfirmationService
  ) {}

  subscriptions = new Subscription();
  messages: Message[] = [];
  filter: any = {};
  @Input() walletTransactions: WalletModelGet[] = [];
  @Input() totalRecords: number = 0;
  @Input() isLockUpdate: boolean;
  isLoading: boolean = false;
  query: BaseParams;
  overlayPanel: boolean;
  search: OrderSearchModel = new OrderSearchModel();
  dataUser: UserProfile[] = [];
  walletId: string;
  walletAmount;
  isLoadingDelete: boolean;
  message: string = MessageStatic.NotFound;
  rowsPerPage = this._pagingHelper.rowsPersPage();
  isLoadingActivate: boolean;
  loadingGetInfo: boolean;
  textHeader: string;
  globalIsSubmited: boolean = false;
  companyId: string;
  first: number = 0;
  last: number = 0;

  ngOnDestroy(): void {
    this.subscriptions.unsubscribe();
  }

  ngOnInit(): void {
    if (!stringIsEmptyOrNull(this._activatedroute.snapshot.params['company'])) {
      this.companyId = this._activatedroute.snapshot.params['company'];
    }
    this.query = new BaseParams();
    this.getData();
    this.cdr.detectChanges();
  }

  userGetInfo() {
    this.loadingGetInfo = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(Address.ComopanyClientGetAll, {}, false)
        .pipe(
          finalize(() => {
            this.loadingGetInfo = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this.dataUser = response.data;
            this.cdr.detectChanges();
          },
          (error) => {}
        )
    );
  }

  goBackToPrevPage() {
    this.location.back();
  }

  getData(event: TableLazyLoadEvent = null): void {
    this.isLoading = true;
    this.cdr.detectChanges();
    this.query = new BaseParams();
    if (this.search.status) this.search.status = Number(this.search.status);
    Object.assign(this.query, this.search);
    this.query = this._pagingHelper.requestParamsSet(event, this.query);
    this.query.companyId = this.companyId;
    this.subscriptions.add(
      this._commonService
        .put(Address.ComopanyClientGetAll, this.query)
        .pipe(
          finalize(() => {
            this.isLoading = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe((res) => {
          if (res && res.data) {
            this.dataUser = cloneDeep(res.data);
            this.totalRecords = res[ColumnKey.TotalRecordKey]
              ? res[ColumnKey.TotalRecordKey]
              : 0;
            this.first = this._pagingHelper.firstRecord(this.query);
            this.last = this._pagingHelper.lastRecord(
              this.query,
              this.totalRecords
            );

            this.cdr.detectChanges();
          }
        })
    );
  }

  toggle(item: UserProfile) {
    item.overlayDocuments = !item.overlayDocuments;
  }

  userModalSave(
    item: UserProfile = null,
    isSubmited: boolean = false
  ) {
    this.textHeader = 'افزودن کاربر جدید';
    const ref = this.dialogService.open(AddUserModalComponent, {
      header: this.textHeader,
      height: 'auto',
      width: '32%',
      styleClass: 'addUser-modal',
      dismissableMask: true,
      data: {
        user: item,
        isSubmited: isSubmited,
        companyId: this.companyId,
        isLockUpdate: this.isLockUpdate,
      },
    });
    ref.onClose.subscribe((model: any) => {
      if (model) {
        this.getData();
      }
    });
  }

  deleteUser(item: UserProfile) {
    this.confirmationService.confirm({
      message: 'آیا از لغو تخصیص کاربر اطمینان دارید؟',
      header: 'لغو تخصیص به کاربر',
      icon: Icon.Warning,
      accept: () => {
        this.subscriptions.add(
          this._commonService
            .delete(Address.CompanyClientDelete, {
              companyClientId: item.companyClientId,
              companyId: this.companyId,
            })
            .pipe(
              finalize(() => {
                this.isLoadingDelete = false;
                this.cdr.detectChanges();
              })
            )
            .subscribe(
              (res) => {
                this._messageService.success('لغو تخصیص با موفقیت انجام شد.');
                this.getData();
                this.isLoadingDelete = false;
                this.cdr.detectChanges();
              },
              (error: any) => {},
              () => {}
            )
        );
      },
      reject: () => {},
      key: 'confirmRemove',
    });
  }
}
