import {
  ChangeDetectorRef,
  Component,
  ElementRef,
  OnDestroy,
  OnInit,
  ViewChild,
} from '@angular/core';
import { Address } from '@enums/address.enum';
import { response } from '@interfaces/response';
import { CommonService } from '@services/common.service';
import { DynamicDialogConfig, DynamicDialogRef } from 'primeng/dynamicdialog';
import { Subscription, finalize } from 'rxjs';
import { UserSave } from '@models/customerService/userSaveModel';
import { VerifyModel } from 'src/app/core/params/auth/verifyMode';
import { AlertHelper } from '@helpers/alert.helper';
import {
  convertDateToGeorgian,
  convertDateToJalali,
} from '@helpers/dateTime.helper';
import { stringIsEmptyOrNull } from '@helpers/validator.helper';
import { ActivatedRoute } from '@angular/router';
import { CaptchaComponent } from '@components/captcha/captcha.component';
import { toEnglishDigits } from '@helpers/convertToEnglishDigit.helper';
import { AddingUser } from 'src/app/core/params/user/addingUser';
import {
  CountdownComponent,
  CountdownConfig,
  CountdownEvent,
} from 'ngx-countdown';
import { setSessionStorageItem } from '@helpers/sessionStorage.helper';
import { ColumnKey } from '@enums/columnKey.enum';

@Component({
  selector: 'add-user-modal',
  templateUrl: 'addUserModal.component.html',
  styleUrls: ['addUserModal.component.scss'],
})
export class AddUserModalComponent implements OnDestroy, OnInit {
  subscriptions = new Subscription();
  showAddForm: boolean = false;
  isLoadingSaveUser: boolean = false;
  showOtp: boolean = false;
  showInputMobile: boolean = true;
  isLoadingSendMobile: boolean;
  isLoadingSubmitCode: boolean;
  userSaveModel: UserSave = new UserSave();
  verifyModel: VerifyModel = new VerifyModel();
  userMobile: string;
  otpCode: any;
  leftTime: number;
  isLoadingResendCode: boolean = false;
  model: AddingUser = new AddingUser();
  canResend: boolean = false;
  orderOptions = [
    { value: 'newCode', title: 'دریافت کد' },
    { value: 'users', title: 'ثبت اطلاعات کاربر' },
  ];
  configCount: CountdownConfig = {
    leftTime: 0,
    format: 'm:s',
  };
  isActiveCaptcha: boolean = true;

  @ViewChild('captcha') captcha: CaptchaComponent;
  @ViewChild('account') input: ElementRef;
  @ViewChild('cd', { static: false }) private countdown: CountdownComponent;

  constructor(
    private _commonService: CommonService,
    private cdr: ChangeDetectorRef,
    public config: DynamicDialogConfig,
    public ref: DynamicDialogRef,
    private _activatedRoute: ActivatedRoute,
    private _alert: AlertHelper
  ) {
    this.isActiveCaptcha = true;
  }

  ngAfterViewInit() {
    if (this.config.data) {
      this.userSaveModel.isLockUpdate = this.config.data.isLockUpdate;
    }
  }

  ngOnInit() {
    if (this.config.data && this.config.data.companyId) {
      this.userSaveModel.companyId = this.config.data.companyId;
    }
    this.verifyModel.typeVerify = 101;
    if (!stringIsEmptyOrNull(this._activatedRoute.snapshot.params['company'])) {
      this.userSaveModel.companyId =
        this._activatedRoute.snapshot.params['company'];
    }
    this.setNewTimer(this.verifyModel.typeVerify);
    if (this.config.data.user) {
      this.userSaveModel = this.config.data.user;
      this.userSaveModel.isActive = this.config.data.user.isActive;
      this.userSaveModel.birthDateFa = convertDateToJalali(
        this.userSaveModel.birthDate
      );
    }
  }

  ngOnDestroy() {
    this.subscriptions.unsubscribe();
  }

  finishTimer(e: CountdownEvent) {
    if (e.action == 'done') {
      this.canResend = true;
    }
  }

  setNewTimer(type: number) {
    switch (type) {
      case 101:
        this.leftTime = 120;
        break;
      case 102:
        this.leftTime = 240;
        break;
    }
    this.configCount = {
      leftTime: this.leftTime,
      format: 'm:s',
    };
    this.canResend = false;
  }

  reSendCode() {
    this.captchaCondotions();
    if (stringIsEmptyOrNull(this.model.tokenRecaptcha)) {
      this._alert.error('لطفا کپچا را تایید کنید');
      return;
    }
    this.isLoadingResendCode = true;
    this.model.tokenVerify = this.verifyModel.token;
    this.model.token = this.verifyModel.token;
    this.model.address = this.userSaveModel.mobile;
    this.model.typeVerify = 101;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .post(Address.ResendCode, this.model, false)
        .pipe(
          finalize(() => {
            this.isLoadingResendCode = false;
            this.captcha.handleRefreshCaptcha();
            this.cdr.detectChanges();
          })
        )
        .subscribe((response: response) => {
          this._alert.success('کد یکبار مصرف مجددا ارسال شد');
          setTimeout(() => {
            this.setNewTimer(this.verifyModel.typeVerify);
          }, 1000);
          this.model.token = response.data.token;
          setSessionStorageItem(ColumnKey.TokenVerify, response.data.token);
          setSessionStorageItem(ColumnKey.Address, response.data.address);
          setSessionStorageItem(ColumnKey.TypeVerify, response.data.typeVerify);
        })
    );
  }

  onCaptchaResolved(response: any): void {
    console.log('reCAPTCHA v2 Response:', response);
  }
  onCaptchaErrored(e: any) {
    console.warn(`reCAPTCHA error encountered`);
  }

  onOtpChange(otp) {
    this.verifyModel.code = otp;
    if (this.verifyModel.code.length == 5) {
      this.action();
    }
  }

  getSubmitMethod(): any {
    return {
      callSubmitMethod: () => {
        this.sendVerifyCodeForUser();
      },
    };
  }

  captchaCondotions() {
    if (
      (this.captcha &&
        this.captcha.captchaInfo &&
        this.captcha.captchaInfo.captchaId &&
        this.captcha.captchaInfo.captchaCode) ||
      !this.isActiveCaptcha
    ) {
      if (this.isActiveCaptcha) {
        const { captchaCode, captchaId } = this.captcha.captchaInfo;
        this.model.captchaCode = captchaCode;
        this.model.captchaCode = toEnglishDigits(this.model.captchaCode);
        this.model.captchaId = captchaId;
      } else {
        this.model.captchaCode = null;
        this.model.captchaId = null;
      }
    }
  }

  sendVerifyCodeForUser() {
    this.captchaCondotions();
    this.userSaveModel.mobile = this.userMobile;
    this.model.address = this.userMobile;
    if (!this.model.captchaCode || !this.model.captchaId) {
      this._alert.error('کد امنیتی خالی است.');
      return;
    }
    this.isLoadingSendMobile = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .post(Address.CompanyCodeSendInPanel, this.model, false)
        .pipe(
          finalize(() => {
            this.captcha.handleRefreshCaptcha();
            this.isLoadingSendMobile = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            if (response.data) {
              this.showAddForm = true;
              this.showOtp = true;
              this.verifyModel = response.data;
            }
            this.cdr.detectChanges();
          },
          (error) => {}
        )
    );
  }

  submitInfo() {
    this.userSaveModel.birthDate = convertDateToGeorgian(
      this.userSaveModel.birthDateFa
    );
    this.userSaveModel.mobile = this.verifyModel.address;
    this.userSaveModel.code = this.verifyModel.code;
    this.userSaveModel.tokenVerify = this.verifyModel.token;
    this.isLoadingSubmitCode = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .post(Address.ComopanyClientSave, this.userSaveModel)
        .pipe(
          finalize(() => {
            this.isLoadingSubmitCode = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe((response) => {
          if (response) {
            this._alert.success('ثبت کاربر با موفقیت انجام شد.');
            this.ref.close(true);
            this.cdr.detectChanges();
          }
        })
    );
  }

  action() {
    if (
      !stringIsEmptyOrNull(this.userSaveModel.nationalCode) &&
      !stringIsEmptyOrNull(this.userSaveModel.birthDateFa) &&
      !stringIsEmptyOrNull(this.userSaveModel.family) &&
      !stringIsEmptyOrNull(this.userSaveModel.name)
    ) {
      this.submitInfo();
    } else {
      this._alert.error('لطفا از وارد کردن تمامی اطلاعات مطمئن شوید.');
      this.cdr.detectChanges();
    }
  }
}