import { AlertHelper } from './../../../core/helpers/alert.helper';
import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  EventEmitter,
  Input,
  OnChanges,
  OnDestroy,
  OnInit,
  Output,
} from '@angular/core';
import { finalize, Subscription } from 'rxjs';
import { CommonService } from '@services/common.service';
import { Address } from '@enums/address.enum';
import { response } from '@interfaces/response';
import { TranslateService } from '@ngx-translate/core';
import { Icon } from '@enums/iconKey.enum';
import { ConfirmationService } from 'primeng/api';
import { AddBankAccountComponent } from './add-bank-account/addBankAccount.component';
import { DialogService } from 'primeng/dynamicdialog';
import { IBankAccount } from '@interfaces/user/IBankAccount';
import { fadeAnimation } from 'src/app/core/animations/animations';
import { FileUrlGeneratorStore } from '@stores/fileUrlGenerator.store';

@Component({
  selector: 'app-bank-account',
  templateUrl: './bank-account.component.html',
  styleUrls: ['./bank-account.component.scss'],
  changeDetection: ChangeDetectionStrategy.OnPush,
  animations: [fadeAnimation],
  host: { '[@fadeInAnimation]': '' },
})
export class BankAccountComponent implements OnInit, OnChanges, OnDestroy {
  constructor(
    private _alert: AlertHelper,
    private _commonService: CommonService,
    private cdr: ChangeDetectorRef,
    private confirmationService: ConfirmationService,
    private dialogService: DialogService,
    private _translate: TranslateService,
    private _fileUrlGeneratorStore: FileUrlGeneratorStore
  ) { }
  cardNumber: string = null;
  isLoadingGet: boolean = false;
  limitForAddBankingCard: string;
  @Input() data: IBankAccount[] = [];
  @Output() onSave = new EventEmitter();
  dataList: IBankAccount[] = [];
  subscriptions = new Subscription();

  ngOnInit(): void {
    this._translate.get('limitForAddBankingCard').subscribe((response) => {
      this.limitForAddBankingCard = response;
      this.cdr.detectChanges();
    });
  }

  ngOnDestroy() {
    this.subscriptions.unsubscribe();
  }

  ngOnChanges() {
     this._fileUrlGeneratorStore
      .getListWithFileUrlCardBankBehavior()
      .subscribe((res) => {
        if (res != null) {
          this.data = res;
          this.cdr.detectChanges();
        }
      });
  }

  getBankAccount() {
    this.isLoadingGet = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(Address.BankAccountGet, {}, false)
        .pipe(
          finalize(() => {
            this.isLoadingGet = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this.dataList = response.data;
            if (this.dataList != null && this.dataList.length > 0) {
              this.data = this.dataList;
            }
          },
          (error) => { }
        )
    );
  }

  addBankAccount() {
    const ref = this.dialogService.open(AddBankAccountComponent, {
      header: 'افزودن حساب بانکی',
      dismissableMask: true,
    });

    ref.onClose.subscribe((model: any) => {
      if (model) {
        this.isLoadingGet = true;
        this.cdr.detectChanges();
        this._alert.success('حساب بانکی جدید با موفقیت افزوده شد');
        this.getBankAccount();
      }
    });
  }

  remove(item: IBankAccount = null) {
    this.confirmationService.confirm({
      message: 'آیا از حذف کارت بانکی خود مطمئن هستید؟',
      header: 'حذف کارت بانکی',
      icon: Icon.Warning,

      accept: () => {
        item.isLoadingDelete = true;
        this.subscriptions.add(
          this._commonService
            .delete(Address.BankAccountDelete, {
              bankAccountId: item.bankAccountId,
            })
            .pipe(
              finalize(() => {
                item.isLoadingDelete = false;
                this.cdr.detectChanges();
              })
            )
            .subscribe(
              (res) => {
                this._alert.success('کارت بانکی شما با موفقیت حذف شد');
                this.getBankAccount();
                this.cdr.detectChanges();
              },
              (error: any) => { },
              () => { }
            )
        );
      },
      reject: () => { },
      key: 'confirmRemove',
    });
  }

  setAsDefault(item: IBankAccount = null) {
    this.confirmationService.confirm({
      message: 'آیا از پیشفرض شدن این کارت اطمینان دارید؟',
      header: 'انتخاب پیشفرض کارت',
      rejectButtonStyleClass: 'bankCardModal',
      dismissableMask: true,

      accept: () => {
        item.isLoadingDelete = true;
        this.subscriptions.add(
          this._commonService
            .put(Address.BankAccountSetDefault, {
              bankAccountId: item.bankAccountId,
            })
            .pipe(
              finalize(() => {
                item.isLoadingDelete = false;
                this.cdr.detectChanges();
              })
            )
            .subscribe(
              (res) => {
                this._alert.success('درخواست شما انجام شد.');
                this.getBankAccount();
                this.cdr.detectChanges();
              },
              (error: any) => { },
              () => { }
            )
        );
      },
      reject: () => { },
      key: 'confirmRemove',
    });
  }
}