import { AlertHelper } from './../../../core/helpers/alert.helper';
import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  EventEmitter,
  Input,
  OnChanges,
  OnDestroy,
  OnInit,
  Output,
  ViewChild,
} from '@angular/core';
import { isValidEmail, stringIsEmptyOrNull } from '@helpers/validator.helper';
import {
  CountdownConfig,
  CountdownComponent,
  CountdownEvent,
} from 'ngx-countdown';
import { finalize, Subscription } from 'rxjs';
import { response } from '@interfaces/response';
import { UserStore } from '@stores/user.store';
import { CommonService } from '@services/common.service';
import { Address } from '@enums/address.enum';
import { UserProfile } from '@interfaces/user/userProfile';
import { VerifyModel } from 'src/app/core/params/auth/verifyMode';
import { fadeAnimation } from 'src/app/core/animations/animations';
import { CaptchaComponent } from '@components/captcha/captcha.component';
import { AuthStore } from '@stores/AuthSetting.store';

@Component({
  selector: 'app-email-verification',
  templateUrl: './email-verification.component.html',
  styleUrls: ['./email-verification.component.scss'],
  changeDetection: ChangeDetectionStrategy.OnPush,
  animations: [fadeAnimation],
  host: { '[@fadeInAnimation]': '' },
})
export class EmailVerificationComponent implements OnChanges, OnDestroy {
  constructor(
    private _alert: AlertHelper,
    private _commonService: CommonService,
    private _userStore: UserStore,
    private _authSettingStore: AuthStore,
    private cdr: ChangeDetectorRef
  ) {
    this._authSettingStore.isActiveCaptcha.subscribe((res) => {
      if (res != null) {
        this.isActiveCaptcha = res;
        this.cdr.detectChanges();
      }
    });

    this._authSettingStore.isActiveCaptchaBehavior.subscribe((res) => {
      if (res != null) {
        this.isActiveCaptcha = res;
        this.cdr.detectChanges();
      }
    });
  }

  @ViewChild('captcha') captcha: CaptchaComponent;
  isActiveCaptcha: boolean;
  isShow = true;
  model: VerifyModel = new VerifyModel();
  email: string;
  emailOld:string;
  isLoading: boolean = false;
  isLoadingResendCode: boolean = false;
  canResend: boolean = false;
  isShowOtp: boolean = false;
  isEditting: boolean = false;
  setInterval = setInterval;
  isLoadingGet: boolean = false;
  isLoadingVerify: boolean = false;
  otp: string;
  @Output() onSave = new EventEmitter();
  @Input() data: UserProfile = null;
  subscriptions = new Subscription();
  @ViewChild('cd', { static: false }) private countdown: CountdownComponent;

  config: CountdownConfig = {
    leftTime: 0,
    format: 'm:s',
  };

  setNewTimer() {
    this.config = {
      leftTime: 120,
      format: 'm:s',
    };
    this.canResend = false;
    this.cdr.detectChanges();
  }

  finishTimer(e: CountdownEvent) {
    if (e.action == 'done') {
      this.canResend = true;
      this.captcha.handleRefreshCaptcha();
    }
    this.cdr.detectChanges();
  }

  ngOnDestroy() {
    this.subscriptions.unsubscribe();
  }

  ngOnChanges() {
    if (this.data != undefined && this.data != null) {
      this.email = this.data.email;
      this.canResend = false;
      this.cdr.detectChanges();
    }
  }

  editInfo() {
    this.emailOld = this.email;
    this.isEditting = true;
    this.data.email = '';
    this.canResend = true;
    this.cdr.detectChanges();
  }

  cancel() {
    this.isEditting = false;
    this.canResend = false;
    this.isShowOtp = false;
    this.email   = this.emailOld;
    this.data.email = this.emailOld;
    this.cdr.detectChanges();
  }
  userGetInfo() {
    this.isLoadingGet = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(Address.UserGet, {}, false)
        .pipe(
          finalize(() => {
            this.isLoadingGet = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this.data = response.data;
            this.email = this.data.email;
            this._userStore.user.next(response.data);
            this._userStore.userBehavior.next(response.data);
            this.cdr.detectChanges();
          },
          (error) => {}
        )
    );
  }

  verify() {
    if (stringIsEmptyOrNull(this.otp)) {
      this._alert.warning('لطفا کد یکبار مصرف را وارد کنید');
      return;
    }
    this.isLoadingVerify = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .post(
          Address.VerifyInPanel,
          {
            code: this.otp,
            typeVerify: 102,
            tokenVerify: this.model.token,
            address: this.model.address,
          },
          false
        )
        .pipe(
          finalize(() => {
            this.isLoadingVerify = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe((response: response) => {
          this.isShowOtp = false;
          this.canResend = true;
          this.data.email = this.email;
          this.isEditting = false;
          this.onSave.emit(true);
          this._alert.success('ایمیل شما با موفقیت تایید شد');
          this.cdr.detectChanges();
          this.userGetInfo();
        })
    );
  }

  submit() {
    if (stringIsEmptyOrNull(this.email)) {
      this._alert.warning('لطفا ایمیل خود را وارد کنید');
      return;
    }

    if (!isValidEmail(this.email)) {
      this._alert.warning('ایمیل وارد شده صحیح نمی باشد');
      return;
    }

    if (
      this.isActiveCaptcha &&
      (!this.captcha.CaptchaCode ||
        stringIsEmptyOrNull(this.captcha.CaptchaCode.value))
    ) {
      this._alert.warning('لطفا کد امنیتی را وارد کنید');
      return;
    }

    this.isLoading = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .post(
          Address.SendInPanel,
          {
            address: String(this.email),
            typeVerify: 102,
            captchaCode: this.captcha.captchaInfo.captchaCode,
            captchaId: this.captcha.captchaInfo.captchaId,
          },
          false
        )
        .pipe(
          finalize(() => {
            this.isLoading = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe((response: response) => {
          setTimeout(() => {
            this.setNewTimer();
          }, 1000);
          this.isShowOtp = true;
          this.model.token = response.data.token;
          this.model.typeVerify = response.data.typeVerify;
          this.model.address = response.data.address;
          this._alert.success('بزودی ایمیل تایید برای شما ارسال خواهد شد');
          this.cdr.detectChanges();
        },(err)=>{
                     this.captcha.handleRefreshCaptcha();
        })
    );
  }
}
