import { stringIsEmptyOrNull } from './../../../core/helpers/validator.helper';
import { AlertHelper } from '@helpers/alert.helper';
import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  EventEmitter,
  Input,
  OnChanges,
  OnDestroy,
  Output,
} from '@angular/core';
import { Router } from '@angular/router';
import { Address } from '@enums/address.enum';
import { UpdateProfileModel } from '@models/user/UpdateProfileModel';
import { CommonService } from '@services/common.service';
import { ConfirmationService } from 'primeng/api';
import { Subscription } from 'rxjs';
import { finalize } from 'rxjs/operators';
import { response } from '@interfaces/response';
import { genders } from '@helpers/values.helper';
import { UserStore } from '@stores/user.store';
import { cloneDeep } from 'lodash-es';
import { fadeAnimation } from 'src/app/core/animations/animations';
import { UserProfile } from '@interfaces/user/userProfile';
import { isAlphabetic } from '@helpers/isAlphabetic.helper';

@Component({
  selector: 'app-update-profile',
  templateUrl: './update-profile.component.html',
  styleUrls: ['./update-profile.component.scss'],
  changeDetection: ChangeDetectionStrategy.OnPush,
  animations: [fadeAnimation],
  host: { '[@fadeInAnimation]': '' },
})
export class UpdateProfileComponent implements OnDestroy, OnChanges {
  isLoading: boolean = false;
  isLoadingGet: boolean = false;
  subscriptions = new Subscription();
  avatar: UserProfile;
  profileData: any;
  @Output() onSave = new EventEmitter();
  @Input() data: UpdateProfileModel = null;

  userInfo: UserProfile = new UserProfile();
  gender = genders();

  constructor(
    private _commonService: CommonService,
    private _router: Router,
    private confirmationService: ConfirmationService,
    private _alert: AlertHelper,
    private _userStore: UserStore,
    private cdr: ChangeDetectorRef
  ) {
    this._userStore.user.subscribe((res) => {
      if (res != null) {
        this.userInfo = res;
      }
    });
    this._userStore.getUserBehavior().subscribe((res) => {
      if (res != null) {
        this.userInfo = res;
      }
    });

  }

  ngOnDestroy() {
    this.subscriptions.unsubscribe();
  }

  ngOnChanges() {
    if (this.data != undefined && this.data != null)
      this.profileData = this.data;
    this.avatar = this.profileData;
  }

  confirm(alarm: string) {
    this.confirmationService.confirm({
      message: alarm,
      key: 'alarm',
      accept: () => {
        this._router.navigateByUrl('/');
      },
    });
  }

  userGetInfo() {
    this.isLoadingGet = true;
    this.subscriptions.add(
      this._commonService
        .put(Address.UserGet, {}, false)
        .pipe(
          finalize(() => {
            this.isLoadingGet = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this.profileData = cloneDeep(response.data);

            this._userStore.user.next(this.profileData);
            this._userStore.userBehavior.next(this.profileData);

          },
          (error) => { }
        )
    );
  }

  submit() {
    if (stringIsEmptyOrNull(this.profileData.birthDate)) {
      this._alert.warning('لطفا تاریخ تولد خود را وارد کنید');
      return;
    }

    let isAlphabeticName =  isAlphabetic(this.avatar.name.replace(/\s/g, ""));
    let isAlphabeticFamily = isAlphabetic(this.avatar.family.replace(/\s/g, ""));
    if (isAlphabeticName && isAlphabeticFamily) {
      this.isLoading = true;
      if (this.profileData.gender != null)
        this.profileData.gender = Number(this.profileData.gender);
      this.profileData.isLockUpdate = true;
      this.subscriptions.add(
        this._commonService
          .put(Address.UpdateProfile, this.profileData, false)
          .pipe(
            finalize(() => {
              this.isLoading = false;
              this.cdr.detectChanges();
            })
          )
          .subscribe(
            (response: response) => {
              this.userGetInfo();
              this.onSave.emit(true);
              this._alert.success('ثبت اطلاعات با موفقیت انجام شد');
            },
            (error) => { }
          )
      );
    } else {
      this._alert.warning('مقدار عددی در نام و نام خانوادگی پذیرفته نیست');
    }


  }
 

}
