import { CommonService } from '@services/common.service';
import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  EventEmitter,
  Input,
  OnDestroy,
  OnInit,
  Output,
} from '@angular/core';
import { Address } from '@enums/address.enum';
import { finalize, Subscription } from 'rxjs';
import { HttpEvent, HttpEventType } from '@angular/common/http';
import { UserStore } from '@stores/user.store';
import { stringIsEmptyOrNull } from '@helpers/validator.helper';
import { response } from '@interfaces/response';
import { FileUploadService } from '@services/fileUpload.service';
import { MessageService, PrimeNGConfig } from 'primeng/api';
import { fadeAnimation } from 'src/app/core/animations/animations';

@Component({
  selector: 'uploader-avatar',
  templateUrl: 'uploaderAvatar.component.html',
  styleUrls: ['uploaderAvatar.component.scss'],
  changeDetection: ChangeDetectionStrategy.OnPush,
  animations: [fadeAnimation],
  host: { '[@fadeInAnimation]': '' }
})
export class UploaderAvatarComponent implements OnInit, OnDestroy {
  constructor(
    private _commonService: CommonService,
    private cdr: ChangeDetectorRef,
    private _fileUploadService: FileUploadService,
    private _userStore: UserStore,
    private messageService: MessageService,
  
  ) {}
  progress: number = 0;
  @Output() getValueUpload = new EventEmitter();
  @Output() removeValueUpload = new EventEmitter();
  subscriptions = new Subscription();
  file: File;
  @Input() id: string = null;
  @Input() extension: string = ".png,.jpg,.jpeg,.gif,.PNG,.JPG,.JPEG,.jfif,.bmp";
  extensionStore: string;
  @Input() url: string = null;
  isLoading: boolean = false;
  async ngOnInit() {
   
  }
  ngOnDestroy() {
    this.subscriptions.unsubscribe();
  }

  ngOnChanges() {
    if (!stringIsEmptyOrNull(this.url))
      this.extensionStore = String(
        this.url.substring(this.url.lastIndexOf('.') + 1, this.url.length)
      ).toLowerCase();
     
      
  }
  fileChange(event) {
    let fileList: FileList = event.target!=undefined && event.target!=null?event.target.files:event;
    if (fileList.length > 0) {
      this.file = fileList[0];
      this.fileUpload();
    }
  }

  fileUpload() {
    this.isLoading = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._fileUploadService
        .fileUpload(this.file.name, this.file,Address.FileUploadSmallFileImage)
        .subscribe((event: HttpEvent<any>) => {
          switch (event.type) {
            case HttpEventType.Sent:
              break;
            case HttpEventType.ResponseHeader:
              break;
            case HttpEventType.UploadProgress:
              var eventTotal = event.total ? event.total : 0;
              this.progress = Math.round((event.loaded / eventTotal) * 100);
              this.cdr.detectChanges();
              break;
            case HttpEventType.Response:
              if (event.body['data'] != null) {
                this.id = event.body['data'];
                this.getValueUpload.emit(event.body);
                this.updateAvatar(this.id);
                this.cdr.detectChanges();
              }
              setTimeout(() => {
                this.progress = 0;
                this.cdr.detectChanges();
              }, 1500);
          }
        })
    );
  }

  updateAvatar(id: string) {
    this.subscriptions.add(
      this._commonService
        .put(Address.UpdateAvatar, { imageId: id })
        .pipe(
          finalize(() => {
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response) => {
            this.userGetInfo();
            
            this.messageService.add({
              severity: 'success',
              detail:'آپلود عکس پروفایل با موفقیت انجام شد',
          })
            
          },
          (err: any) => {},
          () => {}
        )
    );
  }


  loadFiles(id: string) {
    this.subscriptions.add(
      this._commonService
        .getResponseText(Address.GetFileUrl+id)
        .pipe(
          finalize(() => {
            this.isLoading = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response) => {
            this.url = response;
            this.extensionStore = String(
              this.url.substring(this.url.lastIndexOf('.') + 1, this.url.length)
            ).toLowerCase();
            this.cdr.detectChanges();
          },
          (err: any) => {},
          () => {}
        )
    );
  }

  userGetInfo() {
    this.subscriptions.add(
      this._commonService
        .put(Address.UserGet, {}, false)
        .pipe(
          finalize(() => {
            this.isLoading = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this.url = response.data.fileUrl;
            this.extensionStore = String(
              this.url.substring(this.url.lastIndexOf('.') + 1, this.url.length)
            ).toLowerCase();
            this._userStore.user.next(response.data);
            this._userStore.userBehavior.next(response.data);
          },
          (error) => {}
        )
    );
  }

  remove() {
    this.url = null;
    this.removeValueUpload.emit(this.id);
  }
}
