import { ChangeDetectorRef, Component, OnDestroy, OnInit } from '@angular/core';
import {
  Subscription,
  distinctUntilChanged,
  filter,
  finalize,
  skip,
} from 'rxjs';
import { Router } from '@angular/router';
import { RedirectorHelper } from '@helpers/redirector.helper';
import { stringIsEmptyOrNull } from '@helpers/validator.helper';
import { SocialAuthService } from '@abacritt/angularx-social-login';
import { CommonService } from '@services/common.service';
import { Address } from '@enums/address.enum';
import { setStorageItemCallback } from '@helpers/localstorage.helper';
import { ColumnKey } from '@enums/columnKey.enum';
import { AuthStore } from '@stores/AuthSetting.store';
import { CountryModel } from '@models/orders/CountryModel';
import { ConstantService } from '@services/constant.service';
import { FileUploadService } from '@services/fileUpload.service';

@Component({
  selector: 'loginRegisterWizard',
  templateUrl: './loginRegisterWizard.component.html',
  styleUrls: ['./loginRegisterWizard.component.scss'],
})
export class LoginRegisterWizardComponent implements OnInit, OnDestroy {
  subscriptions = new Subscription();
  url: string = 'login';
  urlRoute: string = 'login';
  indexTab: number = 0;
  isGoogleVerifying: boolean = false;
  isActiveOAuth: boolean;
  isLoadingGetCountry: boolean = false;
  selectedCountry: CountryModel;
  isLodingGetUrl: boolean;
  dataCountry: CountryModel[] = [];
  constructor(
    private _router: Router,
    public _redirectorHelper: RedirectorHelper,
    private authService: SocialAuthService,
    private _commonService: CommonService,
    private cdr: ChangeDetectorRef,
    private _authSettingStore: AuthStore,
    private constantService: ConstantService,
    private _fileUploadService: FileUploadService
  ) {
    if (!stringIsEmptyOrNull(this._router.url) && this._router.url != '/')
      this.url = this._router.url.replace('/', '');
    if (this.url.indexOf('?') > 0) this.urlRoute = this.url.split('?')[0];
    else this.urlRoute = this.url;

    this.handleChangeRoute(this.url);
    this._authSettingStore.isActiveOAuth.subscribe((res) => {
      if (res != null) {
        this.isActiveOAuth = res;
      }
    });

    this._authSettingStore.isActiveOAuthBehavior.subscribe((res) => {
      if (res != null) {
        this.isActiveOAuth = res;
      }
    });
  }

  ngOnInit(): void {
    this.getVerifyOAuth();
    this.getCountry();
    this.subscriptions.add(
      this.constantService.isForeign$
        .pipe(
          distinctUntilChanged(),
          skip(1),
          filter((val) => val !== null)
        )
        .subscribe((newIsForeign) => {
          if (newIsForeign === true) {
            this.getVerifyOAuth();
            this.getCountry();
          }
        })
    );
  }

  getVerifyOAuth() {
    if (this.isActiveOAuth) {
      this.subscriptions.add(
        this.authService.authState.subscribe((user) => {
          if (user != null) {
            this.authService.signOut();

            this.subscriptions.add(
              this._commonService
                .putforlogin(Address.VerifyOAuth, { tokenId: user.idToken })
                .pipe(finalize(() => {}))
                .subscribe(
                  (response: any) => {
                    setStorageItemCallback(
                      ColumnKey.UserToken,
                      {
                        access_token: response.data.access_token,
                        refresh_token: response.data.refresh_token,
                      },
                      () => {
                        this._redirectorHelper.routerHandlerAfterAuth();
                      }
                    );
                  },
                  () => {}
                )
            );
          }
        })
      );
    }
  }

  googleSignin(googleWrapper: any) {
    googleWrapper.click();
    this.cdr.detectChanges();
  }

  ngOnDestroy(): void {
    this.subscriptions.unsubscribe();
  }

  handleChange(e) {
    switch (e.index) {
      case 0:
        this._redirectorHelper.routerHandlerAuth('register');
        break;
      case 1:
        this._redirectorHelper.routerHandlerAuth('login');
        break;
    }
  }

  handleChangeRoute(e: string) {
    if (e.indexOf('?') > 0) e = e.split('?')[0];

    switch (e) {
      case 'register':
        this.indexTab = 0;
        break;
      case 'login':
        this.indexTab = 1;
        break;
      case 'loginByUserName':
        this.indexTab = 1;
        break;
    }
  }

  getCountry() {
    this.isLoadingGetCountry = true;
    this.subscriptions.add(
      this._commonService
        .put(
          Address.CountryGetAll,
          { useInOrder: null, useInRegister: true },
          false
        )
        .pipe(
          finalize(() => {
            this.isLoadingGetCountry = false;
          })
        )
        .subscribe(
          (response) => {
            this.dataCountry = response.data.filter(
              (item) => item.isActive && !stringIsEmptyOrNull(item.prefixPhone)
            );
            
            this._fileUploadService.fileUrlGeneratorPutCountry(
              this.dataCountry,
              'avatarId'
            );
            for (let country of this.dataCountry) {
              if (country.isoCode == 'IRN') {
                this.selectedCountry = country;
              }
            }
          },
          (error) => {}
        )
    );
  }
}
