import {
  AfterViewInit,
  Component,
  OnDestroy,
  OnInit,
  ViewChild,
} from '@angular/core';
import { CommonService } from '@services/common.service';
import { Subscription } from 'rxjs';
import { finalize, filter, pairwise, tap } from 'rxjs/operators';
import { response } from '@interfaces/response';
import { Router, RoutesRecognized } from '@angular/router';
import { Address } from '@enums/address.enum';
import {
  getSessionStorageItem,
  setSessionStorageItem,
} from '@helpers/sessionStorage.helper';
import { ColumnKey } from '@enums/columnKey.enum';
import {
  CountdownComponent,
  CountdownConfig,
  CountdownEvent,
} from 'ngx-countdown';
import { AlertHelper } from '@helpers/alert.helper';
import { VerifyModel } from 'src/app/core/params/auth/verifyMode';
import { setStorageItem, setStorageItemSync } from '@helpers/localstorage.helper';
import { RedirectorHelper } from '@helpers/redirector.helper';
import { stringIsEmptyOrNull } from '@helpers/validator.helper';
import { toEnglishDigits } from '@helpers/convertToEnglishDigit.helper';
import { AuthStore } from '@stores/AuthSetting.store';
import { NgOtpInputComponent } from 'ng-otp-input';
import { TranslateService } from '@ngx-translate/core';
import { TypeVerify } from '@enums/typeVerify.enum';
@Component({
  selector: 'verify',
  templateUrl: './verify.component.html',
  styleUrls: ['./verify.component.scss'],
})
export class VerifyComponent implements OnInit, OnDestroy, AfterViewInit {
  isLoading: boolean = false;
  isLoadingResendCode: boolean = false;
  isLoadingResendCodeCall: boolean = false;
  isLoadingResendCodeEmail: boolean = false;
  isLoadingResendCodeOnline: boolean = false;
  canResend: boolean = false;
  canResendCall: boolean = false;
  canResendOnline: boolean = false;
  leftTime: number;
  leftTimeCall: number;
  leftTimeOnline: number;
  otp: string;
  isActiveRecaptcha: boolean;
  setInterval = setInterval;
  subscriptions = new Subscription();
  model: VerifyModel = new VerifyModel();
  @ViewChild('cd', { static: false }) countdown: CountdownComponent;
  @ViewChild('otpInputRef', { static: false })
  otpInputRef!: NgOtpInputComponent;
  showHelpLogin: boolean;
  isActiveTTS: boolean;
  startSecondTimer: boolean = false;
  secondTimerRunning: boolean = false;
  isActiveBotTelegram: boolean;
  isActiveWhatsapp: boolean;
  isActiveAuthenticationMobile: boolean;
  parallelSendingWhatsappLogin: boolean;
  isActiveSms: boolean;
  parallelSendingTelegramLogin: boolean;
  codeSentToTelegram: boolean = false;
  codeSentToWhatsapp: boolean = false;
  optionsTypeSendCodeToMobile = [
    { label: 'واتساپ', value: 'whatsapp', icon: 'pi pi-whatsapp' },
  ];
  typeSendCodeToMobile: any = {};
  checkTimerInterval: any;
  config: CountdownConfig = {
    leftTime: 120,
    format: 'm:s',
  };

  configSecond: CountdownConfig = {
    leftTime: 59,
    format: 'm:s',
  };

  configOnline: CountdownConfig = {
    leftTime: 120,
    format: 'm:s',
  };

  setNewTimer(type: number, leftTimeForCall: number = null) {
    switch (type) {
      case 101:
        this.leftTime = 120;
        if (this.model.prefixPhone != '+98') {
          this.leftTimeOnline = 120;
        }
        if (leftTimeForCall) {
          this.leftTimeCall = leftTimeForCall;
        } else {
          this.leftTimeCall = 60;
        }
        break;
      case 103:
        this.leftTime = 60;
        this.leftTimeCall = 60;
        break;
      case 102:
        this.leftTime = 240;
        break;
    }

    this.config = {
      leftTime: this.leftTime,
      format: 'm:s',
    };

    this.configSecond = {
      leftTime: this.leftTimeCall,
      format: 'm:s',
    };

    this.configOnline = {
      leftTime: this.leftTimeOnline,
      format: 'm:s',
    };

    this.canResend = false;
    this.canResendCall = false;
    this.canResendOnline = false;
  }

  updateSendOptions() {
    this.optionsTypeSendCodeToMobile = [];

    if (this.isActiveBotTelegram) {
      this.optionsTypeSendCodeToMobile.push({
        label: 'تلگرام',
        value: 'telegram',
        icon: 'pi pi-telegram',
      });
    }

    if (
      this.isActiveWhatsapp &&
      !this.parallelSendingWhatsappLogin &&
      this.isActiveAuthenticationMobile &&
      this.isActiveSms
    ) {
      this.optionsTypeSendCodeToMobile.push({
        label: 'واتساپ',
        value: 'whatsapp',
        icon: 'pi pi-whatsapp',
      });
    }
  }

  onOtpChange(otp: string): void {
    this.model.code = otp;

    if (otp?.length === 5) {
      this.model.code = toEnglishDigits(otp);
      this.submit();
    }
  }

  finishTimer(e: CountdownEvent) {
    if (e.action == 'done') {
      this.canResend = true;
    }
  }

  finishTimerCall(e: CountdownEvent) {
    if (e.action == 'done') {
      this.canResendCall = true;
    }
  }

  finishTimerOnline(e: CountdownEvent) {
    if (e.action == 'done') {
      this.canResendOnline = true;
    }
  }

  constructor(
    private _commonService: CommonService,
    private _router: Router,
    private _alert: AlertHelper,
    private _redirectorHelper: RedirectorHelper,
    private _authSettingStore: AuthStore,
    private _translateService: TranslateService
  ) {
    this._authSettingStore.isActiveRecaptchahBehavior.subscribe((res) => {
      if (res != null) {
        this.isActiveRecaptcha = res;
      }
    });
    this._authSettingStore.sendCodeToTelegramBehavior.subscribe((res) => {
      if (res != null) {
        this.codeSentToTelegram = res;
      }
    });

    this._authSettingStore.sendCodeToWhatsappBehavior.subscribe((res) => {
      if (res != null) {
        this.codeSentToWhatsapp = res;
      }
    });

    this._authSettingStore.showHelpLoginBehavior.subscribe((res) => {
      if (res != null) {
        this.showHelpLogin = res;
      }
    });

    this._authSettingStore.isActiveTTSBehavior.subscribe((res) => {
      if (res != null) {
        this.isActiveTTS = res;
      }
    });

    this._authSettingStore.isActiveBotTelegramBehavior.subscribe((res) => {
      if (res != null) {
        this.isActiveBotTelegram = res;
        this.updateSendOptions();
      }
    });

    this._authSettingStore.isActiveWhatsappBehavior.subscribe((res) => {
      if (res != null) {
        this.isActiveWhatsapp = res;
        this.updateSendOptions();
      }
    });

    this._authSettingStore.isActiveAuthenticationMobileBehavior.subscribe(
      (res) => {
        if (res != null) {
          this.isActiveAuthenticationMobile = res;
          this.updateSendOptions();
        }
      }
    );

    this._authSettingStore.parallelSendingWhatsappLoginBehavior.subscribe(
      (res) => {
        if (res != null) {
          this.parallelSendingWhatsappLogin = res;
          this.updateSendOptions();
        }
      }
    );

    this._authSettingStore.isActiveSmsBehavior.subscribe((res) => {
      if (res != null) {
        this.isActiveSms = res;
        this.updateSendOptions();
      }
    });

    this._authSettingStore.parallelSendingTelegramLoginBehavior.subscribe(
      (res) => {
        if (res != null) {
          this.parallelSendingTelegramLogin = res;
          this.updateSendOptions();
        }
      }
    );

    this.subscriptions = this._router.events
      .pipe(
        filter((evt: any) => evt instanceof RoutesRecognized),
        tap((event) => { }),
        pairwise()
      )
      .subscribe((events: RoutesRecognized[]) => { });
  }

  ngOnInit(): void {
    this.model.token = getSessionStorageItem(ColumnKey.TokenVerify);
    this.model.typeVerify = Number(getSessionStorageItem(ColumnKey.TypeVerify));
    this.setNewTimer(this.model.typeVerify, 59);
    this.model.address = getSessionStorageItem(ColumnKey.Address);
    this.model.prefixPhone = getSessionStorageItem(ColumnKey.PrefixPhone);
    this.model.countryId = getSessionStorageItem(ColumnKey.countryId);
  }

  ngAfterViewInit(): void {
    this.checkTimerInterval = setInterval(() => {
      if (this.countdown?.left <= 121000 && !this.startSecondTimer) {
        this.startSecondTimer = true;
      }
    }, 1000);
  }

  ngOnDestroy(): void {
    this.subscriptions.unsubscribe();
    if (this.checkTimerInterval) {
      clearInterval(this.checkTimerInterval);
    }
    this.subscriptions.unsubscribe();
  }

  onCaptchaResolved(response: any): void {
    console.log('reCAPTCHA v2 Response:', response);
  }
  onCaptchaErrored(e: any) {
    console.warn(`reCAPTCHA error encountered`);
  }

  reSendCode(code = null) {
    setTimeout(() => {
      this.otpInputRef?.setValue('');
    }, 50);
    if (code) {
      this.model.typeVerify = code;
    }
    if (
      this.typeSendCodeToMobile &&
      this.typeSendCodeToMobile.value == 'whatsapp'
    ) {
      this.model.sendCodeToWhatsApp = true;
      this.model.sendCodeToTelegram = false;
    }
    if (this.typeSendCodeToMobile.value == 'telegram') {
      this.model.sendCodeToTelegram = true;
      this.model.sendCodeToWhatsApp = false;
    }
    if (
      stringIsEmptyOrNull(this.model.tokenRecaptcha) &&
      this.isActiveRecaptcha
    ) {
      this._alert.error(
        this._translateService.translations[this._translateService.currentLang][
        'verifyCaptcha'
        ]
      );
      return;
    }
    switch (code) {
      case 101:
        this.isLoadingResendCode = true;
        if (this.model.prefixPhone != '+98') {
          this.isLoadingResendCodeOnline = true;
        }
        break;
      case 103:
        this.isLoadingResendCodeCall = true;
        break;
      case 102:
        this.isLoadingResendCodeEmail = true;
        break;
    }
    this.subscriptions.add(
      this._commonService
        .post(Address.ResendCode, this.model, false)
        .pipe(
          finalize(() => {
            switch (code) {
              case 101:
                this.isLoadingResendCode = false;
                if (this.model.prefixPhone != '+98') {
                  this.isLoadingResendCodeOnline = false;
                }
                break;
              case 103:
                this.isLoadingResendCodeCall = false;
                break;
              case 102:
                this.isLoadingResendCodeEmail = false;
                break;
            }
          })
        )
        .subscribe(
          (response: response) => {
            this._alert.success(
              this._translateService.translations[
              this._translateService.currentLang
              ]['verifyCodeWasSentAgain']
            );
            setTimeout(() => {
              this.setNewTimer(this.model.typeVerify);
            }, 1000);
            this.model.token = response.data.token;
            setSessionStorageItem(ColumnKey.TokenVerify, response.data.token);
            setSessionStorageItem(ColumnKey.Address, response.data.address);
            setSessionStorageItem(
              ColumnKey.TypeVerify,
              response.data.typeVerify
            );
          },
          (error) => {
            this._router.navigate(['/verify']);
          }
        )
    );
  }

  submit(): void {
    if (this.isLoading || !this.model.code) return;

    this.isLoading = true;

    this._commonService
      .post(Address.Verify, this.model, false)
      .pipe(finalize(() => (this.isLoading = false)))
      .subscribe({
        next: (res) => {
          setStorageItemSync(ColumnKey.UserToken, {
            access_token: res.data.access_token,
            refresh_token: res.data.refresh_token,
          });
          this._redirectorHelper.routerHandlerAfterAuth();
        },
        error: () => { },
      });
  }

  back(editType: 'email' | 'mobile', oldValue: string, typeVerify: number) {
  this._authSettingStore.sendCodeToWhatsappBehavior.next(false);
  this._authSettingStore.sendCodeToTelegramBehavior.next(false);
  this._router.navigate(['/login'], {
    state: { method: editType, address: oldValue, typeVerify: typeVerify },
  });
}
}
