import {
  Component,
  ViewEncapsulation,
  OnDestroy,
  ChangeDetectorRef,
  AfterViewInit,
} from '@angular/core';
import { finalize, Subscription } from 'rxjs';
import { MenuItem } from 'primeng/api/public_api';
import { ActivatedRoute, Router } from '@angular/router';
import { BaseParams } from 'src/app/core/params/user/baseParams';
import { ITransactionSuccess } from '@interfaces/orders/iTransactionSuccess';
import { Address } from '@enums/address.enum';
import { CommonService } from '@services/common.service';
import { stringIsEmptyOrNull } from '@helpers/validator.helper';
import { getStorageItem } from '@helpers/localstorage.helper';
import { ColumnKey } from '@enums/columnKey.enum';
import { response } from '@interfaces/response';

@Component({
  selector: 'callback-payment',
  templateUrl: 'callback-payment.component.html',
  encapsulation: ViewEncapsulation.None,
})
export class CallBackPaymentComponent implements OnDestroy, AfterViewInit {
  constructor(
    private _router: Router,
    private cdr: ChangeDetectorRef,
    private _commonService: CommonService,
    private _activatedRoute: ActivatedRoute
  ) {
    if (
      this._activatedRoute.snapshot.url[0].path == 'success' ||
      this._activatedRoute.snapshot.url[0].path == 'success-wallet'
    ) {
      this.status = 'OK';
    } else {
      this.status = 'Failed';
    }
    
  const paidCodeStr = getStorageItem(ColumnKey.PaidCode);
  const callbackStr = getStorageItem(ColumnKey.CallbackUrl);

  this.paidCode = paidCodeStr ? JSON.parse(paidCodeStr) : null;
  this.callback = callbackStr ? JSON.parse(callbackStr) : null;
  }
  
  status: string;
  paidCode: string;
  callback: string;
  query: BaseParams = new BaseParams();
  subscriptions = new Subscription();
  menuItems: MenuItem[];
  showSide = false;
  data: ITransactionSuccess;
  isLoading: boolean = false;
  ngOnDestroy() {
    this.subscriptions.unsubscribe();
  }
  ngAfterViewInit() {
    if (!stringIsEmptyOrNull(this.paidCode)) {
      this.updateTransaction();
    }
  }
  updateTransaction() {
    this.isLoading = true;
    this.cdr.detectChanges();
    let url = Address.TransactionUpdatePayWalletZarinLink;
    if (this._activatedRoute.snapshot.url[0].path == 'success') {
      url = Address.TransactionUpdatePayZarinLink;
    }
    this.subscriptions.add(
      this._commonService
        .put(url, { purchaseId: this.paidCode, status: this.status }, false)
        .pipe(
          finalize(() => {
            this.isLoading = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this.data = response.data;
            this.cdr.detectChanges();
          },
          (error) => {
            this.status = 'Failed';
          }
        )
    );
  }

  hide() {
    this.showSide = false;
  }
  goToOrder() {
    if (stringIsEmptyOrNull(this.callback))
      this._router.navigateByUrl('/dashboard/orders/invoices');
    else {
      window.location.href = this.callback;
    }
  }
}
