import {
  ChangeDetectorRef,
  Component,
  EventEmitter,
  Input,
  NgModule,
  OnInit,
  Output,
  ViewChild,
  ViewRef,
} from '@angular/core';
import { CommonService } from '@services/common.service';
import { Address } from '@enums/address.enum';
import { distinctUntilChanged, filter, finalize, skip } from 'rxjs/operators';
import { Subscription } from 'rxjs';
import {
  FormBuilder,
  FormControl,
  FormsModule,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { AutoUnsubscribe } from 'src/app/core/decorators/AutoUnSubscribe';
import { AlertHelper } from '@helpers/alert.helper';
import { AuthStore } from '@stores/AuthSetting.store';
import { ConstantService } from '@services/constant.service';
import { TranslateModule, TranslateService } from '@ngx-translate/core';
import { PrimeModule } from '@shared/prime.module';
import { CommonModule } from '@angular/common';

@AutoUnsubscribe({ arrayName: 'subscriptions' })
@Component({
  selector: 'app-captcha',
  templateUrl: './captcha.component.html',
})
export class CaptchaComponent implements OnInit {
  isLoadingCaptcha: boolean = true;
  subscriptions: Subscription[] = [];
  imageCaptcha: string;
  captchaId: string;

  CaptchaCode: FormControl;
  CaptchaCodes: string;
  @Output() captcha: EventEmitter<any> = new EventEmitter<any>();
  @Input() fromChangePass: boolean = false;
  @Input() submitByEnter: any;
  isActiveReCaptcha: boolean;
  constructor(
    private _commonService: CommonService,
    private fb: FormBuilder,
    private _messageService: AlertHelper,
    private _authSettingStore: AuthStore,
    private constantService: ConstantService,
    private _translateService: TranslateService,
    private cdr: ChangeDetectorRef
  ) {
    this._authSettingStore.isActiveRecaptcha.subscribe((res) => {
      if (res != null) {
        this.isActiveReCaptcha = res;
        Promise.resolve().then(() => this.safeDetectChanges());
      }
    });

    this._authSettingStore.isActiveRecaptchahBehavior.subscribe((res) => {
      if (res != null) {
        this.isActiveReCaptcha = res;
        Promise.resolve().then(() => this.safeDetectChanges());
      }
    });
  }

  async ngOnInit() {
    this.CaptchaCode = this.fb.control('', [Validators.required]);
    this.handleFetchCaptcha();

    this.subscriptions.push(
      this.constantService.isForeign$
        .pipe(
          distinctUntilChanged(),
          skip(1),
          filter((val) => val !== null)
        )
        .subscribe((newIsForeign) => {
          if (newIsForeign === true) {
            this.handleFetchCaptcha();
          }
        })
    );
  }

  callParentSubmit() {
    if (this.fromChangePass) {
      if (
        this.captchaInfo &&
        (this.captchaInfo.captchaCode != null ||
          this.captchaInfo.captchaCode != '')
      ) {
        this.submitByEnter.callSubmitMethod();
      } else {
        this._messageService.error(
          this._translateService.translations[
            this._translateService.currentLang
          ]['pleaseEnterSecurityCode']
        );
      }
    }
  }

  private safeDetectChanges() {
    if (!(this.cdr as ViewRef)?.destroyed) {
      this.cdr.detectChanges();
    }
  }

  handleFetchCaptcha() {
    this.CaptchaCodes = '';
    this.isLoadingCaptcha = true;
    this.safeDetectChanges();

    this.subscriptions.push(
      this._commonService
        .get(Address.Captcha)
        .pipe(
          finalize(() => {
            this.isLoadingCaptcha = false;
            this.safeDetectChanges();
          })
        )
        .subscribe({
          next: (response) => {
            const img = response?.data?.imageCaptcha;
            const id = response?.data?.captchaId;
            if (img && id) {
              this.imageCaptcha = 'data:image/png;base64,' + img;
              this.captchaId = id;
            } else {
            }
            this.safeDetectChanges();
          },
          error: (err) => {
            this._messageService.error('خطا در دریافت تصویر امنیتی');
          },
        })
    );
  }

  @ViewChild('CaptchaCodes') inputName;

  handleRefreshCaptcha() {
    this.inputName.nativeElement.value = '';
    this.isLoadingCaptcha = true;
    this.cdr.detectChanges();
    this.subscriptions.push(
      this._commonService
        .get(Address.RefreshCaptcha, {
          oldCaptchaId: this.captchaId,
          isRefresh: true,
        })
        .pipe(
          finalize(() => {
            this.isLoadingCaptcha = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe((response) => {
          this.imageCaptcha =
            'data:image/png;base64,' + response.data.imageCaptcha;
          this.captchaId = response.data.captchaId;
          this.cdr.detectChanges();
        })
    );
  }

  get captchaInfo() {
    return this.CaptchaCode.value && this.captchaId
      ? { captchaCode: this.CaptchaCode.value, captchaId: this.captchaId }
      : null;
  }
}

@NgModule({
  declarations: [CaptchaComponent],
  imports: [
    CommonModule,
    FormsModule,
    ReactiveFormsModule,
    PrimeModule,
    TranslateModule,
  ],
  exports: [CaptchaComponent],
})
export class CaptchaModule {}
