import { Directive, ElementRef, HostListener, Input } from '@angular/core';
import { AlertHelper } from '@helpers/alert.helper';

@Directive({
  selector: '[characterType]',
})
export class CharacterTypeDirective {
  constructor(private el: ElementRef, private _alertHelper: AlertHelper) {}

  @Input() characterType: 'en' | 'fa';
  private warned = false;
  private arabic = /[\u0600-\u06FF]/;

  @HostListener('input')
  onInput() {
    const value = this.el.nativeElement.value as string;

    if (this.characterType === 'en' && this.arabic.test(value)) {
      this._alertOnce('لطفا فقط از حروف انگلیسی استفاده کنید');
      this.removeInvalidChars('en');
    } else if (this.characterType === 'fa' && /[A-Za-z0-9]/.test(value)) {
      this._alertOnce('لطفا فقط از حروف فارسی استفاده کنید');
      this.removeInvalidChars('fa');
    } else {
      this.warned = false;
    }
  }

  @HostListener('paste', ['$event'])
  onPaste(event: ClipboardEvent) {
    const pastedText = event.clipboardData?.getData('text') ?? '';

    if (this.characterType === 'en' && this.arabic.test(pastedText)) {
      this._alertOnce('لطفا فقط از حروف انگلیسی استفاده کنید');
      event.preventDefault();
    }

    if (this.characterType === 'fa' && /[A-Za-z0-9]/.test(pastedText)) {
      this._alertOnce('لطفا فقط از حروف فارسی استفاده کنید');
      event.preventDefault();
    }
  }

  private removeInvalidChars(type: 'en' | 'fa') {
    let value = this.el.nativeElement.value as string;
    if (type === 'en') {
      value = value.replace(this.arabic, '');
    } else {
      value = value.replace(/[A-Za-z0-9]/g, '');
    }
    this.el.nativeElement.value = value;
  }

  private _alertOnce(message: string) {
    if (!this.warned) {
      this._alertHelper.warning(message);
      this.warned = true;
    }
  }
}