import {
  Directive,
  ElementRef,
  HostListener,
  forwardRef,
  Renderer2,
  OnInit,
  Input,
} from '@angular/core';
import { ControlValueAccessor, NG_VALUE_ACCESSOR } from '@angular/forms';

@Directive({
  selector: '[appNumberFormat]',
  providers: [
    {
      provide: NG_VALUE_ACCESSOR,
      useExisting: forwardRef(() => NumberFormatDirective),
      multi: true,
    },
  ],
})
export class NumberFormatDirective implements ControlValueAccessor, OnInit {
  private onChange: (value: number | string) => void = () => {};
  private onTouched: () => void = () => {};
  @Input() separateThousands: boolean = true;
  @Input() groupBy: 'comma' | 'dash' | 'nationalId' = 'comma';

  constructor(private el: ElementRef, private renderer: Renderer2) {}

  ngOnInit() {
    const initialValue = this.el.nativeElement.value;
    if (initialValue) {
      this.formatAndSetValue(initialValue);
    }
  }

  @HostListener('input', ['$event.target.value'])
  onInput(value: string): void {
    let numericValue = this.convertPersianToEnglish(value).replace(
      /[^\u0030-\u0039\u0660-\u0669\.]/g,
      ''
    );

    if (this.groupBy === 'dash') {
      numericValue = numericValue.substring(0, 16);
    }

    if (this.groupBy === 'nationalId') {
      numericValue = numericValue.replace(/\D/g, '').substring(0, 10);
      this.onChange(numericValue);
    } else {
      const parts = numericValue.split('.');
      if (parts.length > 2) {
        numericValue = parts[0] + '.' + parts.slice(1).join('');
      }

      const finalNumber = parseFloat(numericValue) || 0;
      this.onChange(finalNumber);
    }
    this.formatAndSetValue(numericValue);
  }

  private convertPersianToEnglish(value: string): string {
    const persianNumbers = ['۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹'];
    const englishNumbers = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'];
    return value.replace(
      /[۰-۹]/g,
      (char) => englishNumbers[persianNumbers.indexOf(char)]
    );
  }

  private formatAndSetValue(value: string): void {
    if (!value) {
      this.renderer.setProperty(this.el.nativeElement, 'value', '');
      return;
    }

    let formattedValue = value;

    if (this.groupBy === 'dash') {
      const digitsOnly = value.replace(/\D/g, '').substring(0, 16);
      formattedValue = digitsOnly.match(/.{1,4}/g)?.join('-') ?? digitsOnly;
    } else if (this.groupBy == 'nationalId') {
      formattedValue = value.replace(/\D/g, '').substring(0, 10);
    } else {
      const parts = value.split('.');
      let integerPart = parts[0].replace(/\D/g, '');
      if (this.separateThousands) {
        integerPart = integerPart.replace(/\B(?=(\d{3})+(?!\d))/g, ',');
      }
      let decimalPart = parts.length > 1 ? '.' + parts[1] : '';
      formattedValue = integerPart + decimalPart;
    }

    this.renderer.setProperty(this.el.nativeElement, 'value', formattedValue);
  }

  writeValue(value: any): void {
    if (value !== undefined && value !== null) {
      this.formatAndSetValue(value.toString());
    }
  }

  registerOnChange(fn: any): void {
    this.onChange = fn;
  }

  registerOnTouched(fn: any): void {
    this.onTouched = fn;
  }

  setDisabledState(isDisabled: boolean): void {
    this.renderer.setProperty(this.el.nativeElement, 'disabled', isDisabled);
  }
}
