import { Injectable, isDevMode } from '@angular/core';
import {
  HttpRequest,
  HttpHandler,
  HttpInterceptor,
  HttpEvent,
  HttpErrorResponse,
} from '@angular/common/http';
import { Observable, of, throwError } from 'rxjs';
import { Router } from '@angular/router';
import { catchError, finalize } from 'rxjs/operators';
import {
  getStorageItem,
  removeStorageItem,
  setStorageItem,
} from './../helpers/localstorage.helper';
import { MessageService } from 'primeng/api';
import { stringIsEmptyOrNull } from './../helpers/validator.helper';
import { ColumnKey } from '../enums/columnKey.enum';
import { CommonService } from '@services/common.service';
import { Address } from '@enums/address.enum';
import { response } from '@interfaces/response';
import { getPath } from '@helpers/url.helper';

@Injectable()
export class AuthErrorInterceptor implements HttpInterceptor {
  constructor(
    private router: Router,
    private messageService: MessageService,
    private _commonService: CommonService
  ) {}
  private isRefresh = false;
  handleRefreshToken(request: HttpRequest<any>, next: HttpHandler) {
    if (this.isRefresh == false) {
      var refreshToken = JSON.parse(
        getStorageItem(ColumnKey.UserToken)
      ).refresh_token;
      if (!stringIsEmptyOrNull(refreshToken)) {
        this._commonService
          .put(Address.RefreshToken, { refreshToken: refreshToken }, false)
          .pipe(finalize(() => {}))
          .subscribe(
            (response: response) => {
              setStorageItem(ColumnKey.UserToken, {
                access_token: response.data.access_token,
                refresh_token: response.data.refresh_token,
              });
              window.location.reload();
              return next.handle(request);
            },
            (error) => {
              this.router.navigate(['/login'], {
                queryParams: { returnUrl: getPath() },
              });
              return throwError(error);
            }
          );
      }
    }
  }

  intercept(
    request: HttpRequest<unknown>,
    next: HttpHandler
  ): Observable<HttpEvent<unknown>> {
    return next.handle(request).pipe(
      catchError((error: HttpErrorResponse) => {
        const skipError = request.headers.has('X-Skip-Error');
        if ([0].includes(error.status) && !skipError) {
          this.router.navigateByUrl('/errors/server-error');
          return throwError(error);
        }
        if ([460, 464, 465].includes(error.status)) {
          removeStorageItem(ColumnKey.UserToken);
          this.router.navigate(['/login'], {
            queryParams: { returnUrl: getPath() },
          });
          return throwError(error);
        }
        if ([461].includes(error.status)) {
          this.handleRefreshToken(request, next);
        }
        if ([462].includes(error.status)) {
          this.router.navigateByUrl('/errors/access-denied');
          return throwError(error);
        }

        if ([463].includes(error.status)) {
          this.messageService.add({
            severity: 'warn',
            summary: '',
            detail: error.error.message,
          });

          return throwError(error);
        }

        if (
          [
            400, 470, 471, 472, 473, 474, 475, 476, 481, 482, 501, 562, 561,
            560, 504, 466,
          ].includes(error.status)
        ) {
          this.messageService.add({
            severity: 'error',
            summary: '',
            detail: error.error.message,
          });
          return throwError(error);
        }

        if ([467].includes(error.status)) {
          this.router.navigateByUrl('/errors/offline-mode');

          return throwError(error);
        }
        if ([469].includes(error.status)) {
          this.router.navigateByUrl('/errors/banned-mode-1');

          return throwError(error);
        }
        if ([468].includes(error.status)) {
          this.router.navigateByUrl('/errors/banned-mode-2');

          return throwError(error);
        }
        if ([500].includes(error.status)) {
          this.router.navigateByUrl('server-error');

          return throwError(error);
        }

        if ([503, 502, 401, 403].includes(error.status)) {
          this.messageService.add({
            severity: 'error',
            summary: '',
            detail:
              'خطا در ارتباط با سرور بوجود آمده است ، در صورت تکرار مراتب را به پشتیبانی سیستم اطلاع دهید.',
          });

          return throwError(error);
        }

        if (error.status == 429) {
          this.messageService.add({
            severity: 'error',
            summary: '',
            detail:
              'تعداد درخواست‌های شما بیش از حد مجاز است. لطفاً کمی صبر کنید و دوباره تلاش کنید.',
          });
        }

        return throwError(error);
      })
    );
  }
}
