import { Pipe, PipeTransform } from '@angular/core';

@Pipe({
  name: 'enDigits',
  pure: true
})
export class EnDigitsPipe implements PipeTransform {
  private readonly digitMap: { [k: string]: string } = {
    '۰': '0', '۱': '1', '۲': '2', '۳': '3', '۴': '4',
    '۵': '5', '۶': '6', '۷': '7', '۸': '8', '۹': '9',
    '٠': '0', '١': '1', '٢': '2', '٣': '3', '٤': '4',
    '٥': '5', '٦': '6', '٧': '7', '٨': '8', '٩': '9'
  };

  transform(value: number | string | null | undefined, digitsPattern: string = '1.0-0'): string {
    if (value === null || value === undefined || value === '') return '';

    if (typeof value === 'number' && !isNaN(value)) {
      return this.formatNumber(value, digitsPattern);
    }

    let s = String(value);
    s = s.replace(/[\u06F0-\u06F9\u0660-\u0669]/g, (ch) => this.digitMap[ch] ?? ch);
    s = s.replace(/[\u066C\u066B\u00A0\u200F\u200E\u202F]/g, '');
    s = s.replace(/\u060C/g, ',');
    s = s.trim();
    const normalized = s.replace(/,/g, '');
    const num = Number(normalized);

    if (!isNaN(num)) {
      return this.formatNumber(num, digitsPattern);
    }
    return s;
  }

  private formatNumber(n: number, digitsPattern: string) {
    try {
      const options = this.parseDigitsPattern(digitsPattern);
      return new Intl.NumberFormat('en-US', options).format(n);
    } catch (e) {
      return String(n);
    }
  }

  private parseDigitsPattern(p: string) {
    const match = (p || '1.0-0').match(/^\d+\.(\d+)-(\d+)$/);
    if (!match) {
      return { minimumFractionDigits: 0, maximumFractionDigits: 0 };
    }
    const min = parseInt(match[1], 10);
    const max = parseInt(match[2], 10);
    return { minimumFractionDigits: min, maximumFractionDigits: max };
  }
}