import { Injectable } from '@angular/core';
import { HttpClient, HttpHeaders } from '@angular/common/http';
import { environment } from '../../../environments/environment';
import { BehaviorSubject } from 'rxjs';
@Injectable()
export class ConstantService {
  constructor(private http: HttpClient) {}
  private isForeignSubject = new BehaviorSubject<boolean | null>(null);
  public isForeign$ = this.isForeignSubject.asObservable();
  private config: any = null;
  get captchaRoot() {
    return this.getProperty('captcha');
  }
  get apiRoot() {
    return this.getProperty('apiRoot');
  }
  get apiSignalR() {
    return this.getProperty('apiSignalR');
  }
  get fileManager() {
    return this.getProperty('fileManager');
  }
  get logManager() {
    return this.getProperty('logManager');
  }
  get printApi() {
    return this.getProperty('printApi');
  }

  load(): Promise<any> {
    const version = Date.now();
    const filejson = environment.production
      ? `connect-server-production.json?v=${version}`
      : `connect-server-development.json?v=${version}`;

    return this.http
      .get(`./constant/${filejson}`)
      .toPromise()
      .then((config: any) => {
        this.config = config;
        this.isForeignSubject.next(null);

        setTimeout(() => {
          this.getLocation(
            this.getProperty('locationApi'),
            this.getProperty('locationApi2'),
            this.getProperty('geoLocationKey')
          )
            .then((location) => {
              const countryCode = location?.countryCode ?? 'IR';

              if (countryCode !== 'IR' && this.config['externalApi']) {
                this.config['apiRoot'] = this.config['externalApi'];
              }
              if (countryCode !== 'IR' && this.config['externalCaptcha']) {
                this.config['captcha'] = this.config['externalCaptcha'];
              }
              if (countryCode !== 'IR' && this.config['externalFileManager']) {
                this.config['fileManager'] = this.config['externalFileManager'];
              }
              setTimeout(() => {
                if (location?.newInquiry) {
                  if (location.countryCode !== 'IR') {
                    this.isForeignSubject.next(true);
                  } else {
                    this.isForeignSubject.next(false);
                  }
                }
              }, 10);
            })
            .catch((e) => {
              console.warn('Error getting location:', e);
            });
        }, 0);

        return;
      })
      .catch((err) => {
        console.error('Error loading config:', err);
      });
  }

  private async getLocation(
    apiAddressIPAPI: string,
    apiAddressGeoLocation: string,
    geoLocationKeyAccess: string
  ): Promise<any> {
    const cached = localStorage.getItem('user-location');
    const now = Date.now();

    const suspiciousProviders = [
      'M247',
      'DigitalOcean',
      'Contabo',
      'Hetzner',
      'OVH',
      'Linode',
      'LeaseWeb',
      'Private Layer',
      'Amazon',
      'Google Cloud',
      'Microsoft Azure',
      'Choopa',
      'Vultr',
      'DataCamp',
      'Scaleway',
      'Alibaba',
      'Oracle Cloud',
      'Tencent Cloud',
      'G-Core',
      'NetCup',
      'iFog',
      'Frantech',
      'QuadraNet',
      'InterServer',
      'Time4VPS',
      'Shinjiru',
      'HostHatch',
      'UpCloud',
      'Tor Exit',
      'Torguard',
      'NordVPN',
      'ExpressVPN',
      'Surfshark',
      'ProtonVPN',
      'Windscribe',
      'TunnelBear',
      'CyberGhost',
      'HideMyAss',
      'IPVanish',
      'StrongVPN',
      'Private Internet Access',
      'Mullvad',
      'PureVPN',
      'Cloudflare',
      'Incapsula',
      'Sucuri',
      'Hosting',
      'Data Center',
      'Colocation',
      'VPN',
      'Proxy',
      'Server',
      'Anonymous',
    ];

    const suspiciousASNs = [
      'AS14061',
      'AS24940',
      'AS16276',
      'AS12876',
      'AS20473',
      'AS16509',
      'AS13335',
      'AS14618',
      'AS4837',
      'AS37963',
      'AS45090',
      'AS204957',
      'AS9009',
      'AS207960',
      'AS54600',
      'AS396982',
      'AS212238',
      'AS44477',
      'AS26347',
      'AS12389',
    ];

    const timeoutMs = 3000;
    const withTimeout = <T>(promise: Promise<T>, ms: number): Promise<T> => {
      return Promise.race([
        promise,
        new Promise<never>((_, reject) =>
          setTimeout(() => reject(new Error('timeout')), ms)
        ),
      ]);
    };

    if (cached) {
      try {
        const parsed = JSON.parse(cached);

        if (
          parsed.countryCode &&
          String(parsed.countryCode).toUpperCase() !== 'IR'
        ) {
          localStorage.removeItem('user-location');
        } else {
          parsed['newInquiry'] = false;

          const age = now - parsed._ts;
          const asn = parsed.asn?.toString().toUpperCase() ?? '';
          const org = parsed.org?.toLowerCase() ?? '';
          const isp = parsed.isp?.toLowerCase() ?? '';

          const isSuspicious =
            suspiciousProviders.some(
              (p) =>
                org.includes(p.toLowerCase()) || isp.includes(p.toLowerCase())
            ) ||
            suspiciousASNs.some((sasn) => asn.includes(sasn.toUpperCase()));

          const maxAge = isSuspicious ? 5 * 60 * 1000 : 60 * 60 * 1000;

          if (age < maxAge) {
            return parsed;
          }
        }
      } catch (err) {
        console.warn(
          'Invalid cached user-location value. Clearing cache…',
          err
        );
        localStorage.removeItem('user-location');
      }
    }

    const ipGeolocationUrl =
      apiAddressGeoLocation + `?apiKey=${geoLocationKeyAccess}`;

    try {
      const res: any = await withTimeout(
        this.http
          .get(apiAddressIPAPI, {
            headers: new HttpHeaders({ 'X-Skip-Error': 'true' }),
          })
          .toPromise(),
        timeoutMs
      );

      if (!res || res.success === false) throw new Error('ipwho failed');

      const formatted = {
        countryCode: res.country_code || 'IR',
        isp: res.connection?.isp || '',
        org: res.connection?.organization || '',
        asn: res.connection?.asn || '',
        query: res.ip || '',
        _ts: now,
        newInquiry: true,
      };

      localStorage.setItem('user-location', JSON.stringify(formatted));
      return formatted;
    } catch (e) {
      console.warn(
        'Primary geolocation service failed, trying fallback service.',
        e
      );

      try {
        const res2: any = await withTimeout(
          this.http
            .get(ipGeolocationUrl, {
              headers: new HttpHeaders({ 'X-Skip-Error': 'true' }),
            })
            .toPromise(),
          timeoutMs
        );

        const formatted = {
          countryCode: res2.country_code2 || res2.country_code || 'IR',
          isp: res2.isp || '',
          org: res2.organization || '',
          asn: res2.asn || '',
          query: res2.ip || '',
          _ts: now,
          newInquiry: true,
        };

        localStorage.setItem('user-location', JSON.stringify(formatted));
        return formatted;
      } catch (err) {
        console.error(
          'Could not determine location from both services. Assuming IR.',
          err
        );

        const fallback = { countryCode: 'IR', _ts: now, newInquiry: true };
        localStorage.setItem('user-location', JSON.stringify(fallback));
        return fallback;
      }
    }
  }

  private getProperty(property: string): any {
    if (!this.config) {
      throw new Error(
        'Attempted to access configuration property before configuration data was loaded, please implemented.'
      );
    }

    if (!this.config[property]) {
      throw new Error(`Required property ${property} was not defined within the configuration object. Please double check the
        connect-server.json file`);
    }

    return this.config[property];
  }
}
