import { Injectable } from '@angular/core';
import { CommonService } from './common.service';
import { getOnlinePortPayApi } from '@helpers/values.helper';
import { finalize, Subscription } from 'rxjs';
import { setStorageItem } from '@helpers/localstorage.helper';
import { ColumnKey } from '@enums/columnKey.enum';
import { Address } from '@enums/address.enum';
import { stringIsEmptyOrNull } from '@helpers/validator.helper';
import { MessageService } from 'primeng/api';

@Injectable({
  providedIn: 'root',
})
export class OrderChangeStatusService {
  constructor(
    private _commonService: CommonService,
    private messageService: MessageService
  ) {}

  updateStatusDocument(
    orderType: string,
    orderId: string,
    discountCode: string,
    payType: string,
    onNavigate: (url: string) => void,
    onLoading?: (val: boolean) => void
  ) {
    onLoading?.(true);

    this._commonService
      .put(
        Address.OrderUpdateDocumentStatus,
        {
          orderId,
          code: discountCode,
          paymentMethod: payType,
        },
        false
      )
      .pipe(finalize(() => onLoading?.(false)))
      .subscribe(
        (response: any) => {
          onNavigate(
            '/dashboard/orders/depositDocument/' + orderType + '/' + orderId
          );
        },
        (error) => {
          if (error.status === 486) {
            this.messageService.add({
              severity: 'error',
              summary: '',
              detail:
                'جهت ادامه فرایند همکاران با شما ارتباط خواهند گرفت و نحوه پرداخت را به اطلاع شما خواهیم رساند سپاس از صبر و شکیبایی شما',
            });
          }
        }
      );
  }

  filterValidScenariosByAmount(allScenarios: any[], model: any) {
    console.log(allScenarios, 'allScenariosallScenarios');
    if (!Array.isArray(allScenarios)) return [];
    const finalAmount = model.finalAmount || 0;
    const paidAmount = model.paidAmount || 0;
    const isPaid = paidAmount && paidAmount !== 0;
    return allScenarios
      .filter((item: any) => {
        const minValid =
          !item.minimumAmountPaidToStart ||
          finalAmount >= +item.minimumAmountPaidToStart;
        const maxValid =
          !item.maximumAmountPaidToFinish ||
          finalAmount < +item.maximumAmountPaidToFinish;
        return minValid && maxValid;
      })
      .map((s) => ({ ...s }));
  }

  scenarioHasPayment431(scenario: any): boolean {
    if (!scenario) return false;
    const scalarFields = [
      'paymentMethod',
      'paymentGateWay',
      'onlinePortal',
      'bankPort',
    ];

    for (const f of scalarFields) {
      if (scenario[f] !== undefined && scenario[f] !== null) {
        const v = +scenario[f];
        if (!Number.isNaN(v) && v === 431) return true;
      }
    }

    const arrayFields = ['paymentMethods', 'paymentGateways', 'paymentOptions'];
    for (const f of arrayFields) {
      if (Array.isArray(scenario[f])) {
        for (const el of scenario[f]) {
          const v = +el;
          if (!Number.isNaN(v) && v === 431) return true;
        }
      }
    }

    if (scenario.payment && typeof scenario.payment === 'object') {
      const maybe =
        scenario.payment.id ??
        scenario.payment.paymentMethodId ??
        scenario.payment.methodId;
      if (maybe !== undefined) {
        const v = +maybe;
        if (!Number.isNaN(v) && v === 431) return true;
      }
    }

    return false;
  }

  appendOrUpdateQueryParam(urlStr: string, key: string, value: string): string {
    try {
      const url = new URL(urlStr, globalThis.location?.origin ?? '');
      url.searchParams.set(key, value);
      return url.toString();
    } catch (e) {
      console.error('Invalid URL', e);
      const sep = urlStr.includes('?') ? '&' : '?';
      const regex = new RegExp(`[?&]${key}=[^&]*`);
      if (regex.test(urlStr)) {
        return urlStr.replace(
          regex,
          `${sep}${key}=${encodeURIComponent(value)}`
        );
      } else {
        return `${urlStr}${sep}${key}=${encodeURIComponent(value)}`;
      }
    }
  }

  purchase(
    subscriptions: Subscription,
    model: any,
    dataScenario: any[],
    bankPort: number,
    bankId: string,
    checkBankAccount: boolean,
    discountCode: string,
    onFreeOrder: () => void,
    onRefresh: () => void,
    onLoadingChange: (loading: boolean) => void,
    onDetectChanges: () => void
  ) {
    console.log('received dataScenario', dataScenario);

    const url = getOnlinePortPayApi(bankPort);
    model.onlinePortal = bankPort;
    if (checkBankAccount) {
      model.bankAccountId = bankId;
    } else {
      model.bankAccountId = null;
    }

    if (!stringIsEmptyOrNull(discountCode)) {
      model.code = discountCode;
    }

    const allScenarios = dataScenario;
    const secondScenario = allScenarios.find((s) => s.sortNumber === 2);
    let isSmartPayment = false;
    if (secondScenario && +secondScenario.paymentMethod === 431) {
      isSmartPayment = true;
    } else {
      isSmartPayment = false;
    }

    model.returnUrl = this.appendOrUpdateQueryParam(
      globalThis.location?.href ?? '',
      'isSmartPayment',
      isSmartPayment ? 'true' : 'false'
    );
    model.isSmartPayment = isSmartPayment;
    onLoadingChange(true);
    onDetectChanges();
    subscriptions.add(
      this._commonService
        .post(url, model, false)
        .pipe(finalize(() => {}))
        .subscribe(
          (response: any) => {
            if (response.data.isFree) {
              onFreeOrder();
              onLoadingChange(false);
            } else {
              if (!stringIsEmptyOrNull(response.data.paidCode)) {
                setStorageItem(ColumnKey.PaidCode, response.data.paidCode);
              }
              if (globalThis.location) {
                globalThis.location.href = response.data.redirectUrl;
              }
            }

            onDetectChanges();
          },
          (error) => {
            setTimeout(() => {
              let shouldShowRefresh = false;

              if (error.error.message) {
                const userLocationStr = localStorage.getItem('user-location');
                if (userLocationStr) {
                  try {
                    const userLocation = JSON.parse(userLocationStr);
                    if (
                      userLocation.countryCode &&
                      userLocation.countryCode !== 'IR'
                    ) {
                      localStorage.removeItem('user-location');
                      shouldShowRefresh = true;
                    }
                  } catch {
                    localStorage.removeItem('user-location');
                    shouldShowRefresh = true;
                  }
                }
              }

              onLoadingChange(false);
              onDetectChanges();

              if (shouldShowRefresh) {
                onRefresh();
              }
            }, 100);
          }
        )
    );
  }

  cryptoPayment(
    orderId: string,
    onNavigate: (url: string) => void,
    onLoading?: (val: boolean) => void
  ) {
    onLoading?.(true);
    onNavigate('dashboard/orders/payment-crypto/' + orderId);
    onLoading?.(false);
  }

  updateStatusSetAppointment(
    subscriptions: Subscription,
    orderId: string,
    onFreeOrder: () => void,
    onAfterAppointment: () => void,
    onLoadingChange: (loading: boolean) => void
  ) {
    onLoadingChange(true);
    subscriptions.add(
      this._commonService
        .put(
          Address.OrderUpdateStatusSetAppointment,
          { orderId: orderId },
          false
        )
        .pipe(finalize(() => onLoadingChange(false)))
        .subscribe((response: any) => {
          if (response.data.isFree) {
            onFreeOrder();
          } else {
            onAfterAppointment();
          }
        })
    );
  }

  updateFreeOrder(
    subscriptions: Subscription,
    orderId: string,
    discountCode: string,
    onFreeOrder: () => void,
    onLoadingChange: (loading: boolean) => void
  ) {
    onLoadingChange(true);
    subscriptions.add(
      this._commonService
        .post(
          Address.OrderUpdateFreeOrder,
          { orderId, code: discountCode },
          false
        )
        .pipe(finalize(() => onLoadingChange(false)))
        .subscribe((response: any) => {
          onFreeOrder();
        })
    );
  }

  updateStatusWaiting(
    orderId: string,
    subscriptions: Subscription,
    onStatusWaitingConfirm: () => void,
    onLoadingChange: (loading: boolean) => void
  ) {
    onLoadingChange(true);
    subscriptions.add(
      this._commonService
        .put(Address.OrderUpdateWaitingStatus, { orderId: orderId }, false)
        .pipe(finalize(() => onLoadingChange(false)))
        .subscribe((response: any) => {
          onStatusWaitingConfirm();
        })
    );
  }

  purchaseByWallet(
    subscriptions: Subscription,
    orderId: string,
    discountCode: string,
    onFreeOrder: () => void,
    onWalletConfirm: () => void,
    onLoadingChange: (loading: boolean) => void
  ) {
    onLoadingChange(true);
    subscriptions.add(
      this._commonService
        .post(Address.PayByWalletSave, { orderId, code: discountCode }, false)
        .pipe(finalize(() => onLoadingChange(false)))
        .subscribe((response: any) => {
          if (response.data.isFree) {
            onFreeOrder();
          } else {
            onWalletConfirm();
          }
        })
    );
  }
}
