import { Injectable } from '@angular/core';
import { HubConnection, HubConnectionBuilder} from '@microsoft/signalr'
import { from } from 'rxjs';
import { MessagePackHubProtocol } from '@microsoft/signalr-protocol-msgpack'
import { AlertHelper } from '../helpers/alert.helper';
import { RealTimeStore } from '@stores/realTime.store';
import { ConstantService } from './constant.service';

@Injectable({
  providedIn: 'root'
})
export class SignalrService {

  private hubConnection: HubConnection

  constructor(private _apiAddress:ConstantService,private _realTimeStore:RealTimeStore,private _alert:AlertHelper) { }

  public connect = () => {
    this.startConnection();
    this.addListeners();
  }


  public sendMessageToHub(model: string) {
    var promise = this.hubConnection.invoke("BroadcastAsync", model)
      .then(() => { console.log('message sent successfully to hub'); })
      .catch((err) => console.log('error while sending a message to hub: ' + err));
    return from(promise);
  }

  private getConnection(): HubConnection {
    return new HubConnectionBuilder()
      .withUrl(this._apiAddress.apiSignalR)
      .withHubProtocol(new MessagePackHubProtocol())
      .build();
  }


  private reconnectAttempts = 0;
  private maxReconnectAttempts = 5;
  private startConnection() {
    this.hubConnection = this.getConnection();

  this.hubConnection.start()
    .then(() => {
      console.log('connection started');
      this.reconnectAttempts = 0;
    })
    .catch((err) => {
      console.log('Connection failed: ' + err);
      if (this.reconnectAttempts < this.maxReconnectAttempts) {
        this.reconnectAttempts++;
        const delay = Math.min(5000 * this.reconnectAttempts, 30000);
        setTimeout(() => this.startConnection(), delay);
      } else {
        this._alert.error('خطا در قطع اتصال شبکه لحظه ای رخ داده است');
      }
    });

    this.hubConnection.onclose(() => {
      console.log("Connection closed. Trying to reconnect...");
      if (this.reconnectAttempts < this.maxReconnectAttempts) {
        const delay = Math.min(5000 * this.reconnectAttempts, 30000);
        setTimeout(() => this.startConnection(), delay);
      } else {
        this._alert.error('خطا در قطع اتصال شبکه لحظه ای رخ داده است');
      }
    });
  }

  public closeConnection() {
    this.hubConnection = this.getConnection();

    this.hubConnection.stop()
      .then(() => {
        console.log('connection stoped');
        this.hubConnection=undefined;
      })
      .catch((err) => this._alert.error('خطا در قطع اتصال شبکه لحظه ای رخ داده است'))
  }

  private addListeners() {
    this.hubConnection.on("messageReceivedFromApi", (data: string) => {
      console.log("message received from API Controller")
    })
    this.hubConnection.on("messageReceivedFromHub", (data: string) => {
      this._realTimeStore.realTimeData.next({message:data});
    })
    this.hubConnection.on("newUserConnected", _ => {
      console.log("connected.")
    })
  }
}
