import {
  ChangeDetectorRef,
  Component,
  Input,
  OnInit,
  Output,
  EventEmitter,
  ChangeDetectionStrategy,
  OnDestroy,
  OnChanges,
  SimpleChanges,
} from '@angular/core';
import { Address } from '@enums/address.enum';
import { CommonService } from '@services/common.service';
import { Subscription, finalize } from 'rxjs';
import { TranslateService } from '@ngx-translate/core';
import { cloneDeep } from 'lodash-es';
import { DialogService } from 'primeng/dynamicdialog';
import { AddBankAccountComponent } from '../../user/bank-account/add-bank-account/addBankAccount.component';
import { AlertHelper } from '@helpers/alert.helper';
import { IBankAccount } from '@interfaces/user/IBankAccount';
import { FileUploadService } from '@services/fileUpload.service';

@Component({
  selector: 'paymentGateway',
  templateUrl: 'paymentGateway.component.html',
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class PaymentGatewayComponent implements OnInit, OnDestroy, OnChanges {
  @Input() bankPorts: any;
  @Output() bankPortsChange = new EventEmitter<any>();
  @Input() payType: string;
  @Input() bankId: string = null;
  @Output() bankIdChange = new EventEmitter<string>();
  @Input() dataCheckout: any;
  @Input() allScenarios: any[] = [];
  @Input() checkBankAccount: boolean;
  @Input() checkoutActivations: any = {};
  paymentGatewayList: any[] = [];
  subscriptions = new Subscription();
  bankAccountList: IBankAccount[] = [];
  isLoading: boolean = false;
  lock: boolean = false;
  dataScenario: any = null;
  isLoadingGetFile: boolean = false;

  constructor(
    private cdr: ChangeDetectorRef,
    private _commonService: CommonService,
    private _translateService: TranslateService,
    private _alert: AlertHelper,
    private dialogService: DialogService
  ) {}

  ngOnChanges(changes: SimpleChanges) {
    if (
      changes['allScenarios'] ||
      changes['checkoutActivations'] ||
      changes['dataCheckout']
    ) {
      this.checkForScenario();
    }
  }

  choosePaymentGateWay() {
    if (this.lock) return;
    this.paymentGatewayList = [];
    if (
      !(
        this.checkoutActivations &&
        (this.checkoutActivations.isActiveOnlinePaySingle ||
          this.checkoutActivations.isActiveOnlinePayMultiple)
      )
    ) {
      this.bankPorts = null;
      this.bankPortsChange.emit(this.bankPorts);
      this.cdr.detectChanges();
      return;
    }

    this._translateService
      .get('paymentGateway.data')
      .subscribe((allGateways: any[]) => {
        const activeGateways = allGateways.filter((g) => {
          const val = Number(g.value);
          return (
            (val === 452 && this.checkoutActivations.isActiveJibitOnlinePay) ||
            (val === 453 && this.checkoutActivations.isActiveVandar) ||
            (val === 454 && this.checkoutActivations.isActiveShepaPPG) ||
            (val === 455 && this.checkoutActivations.isActiveZarinpal) ||
            (val === 457 && this.checkoutActivations.isActiveZarinLink) ||
            (val === 458 && this.checkoutActivations.isActiveSepal) ||
            (val === 451 && this.checkoutActivations.isActiveMellat)
          );
        });

        const scenarioPortals = new Set<number>();
        if (this.allScenarios && this.allScenarios.length) {
          const finalAmount = this.dataCheckout?.finalAmount ?? 0;
          const paidAmount = this.dataCheckout?.paidAmount ?? 0;
          const isPaid = paidAmount && paidAmount !== 0;
          const targetSortNumber = isPaid ? 2 : 1;
          this.allScenarios.forEach((s) => {
            if (s.sortNumber === targetSortNumber) {
              const minValid =
                !s.minimumAmountPaidToStart ||
                finalAmount >= +s.minimumAmountPaidToStart;
              const maxValid =
                !s.maximumAmountPaidToFinish ||
                finalAmount < +s.maximumAmountPaidToFinish;
              if (minValid && maxValid && s.onlinePortal) {
                scenarioPortals.add(Number(s.onlinePortal));
              }
            }
          });
        }

        if (scenarioPortals.size > 0) {
          this.paymentGatewayList = activeGateways.filter(
            (pg) => scenarioPortals.has(Number(pg.value)) || true // همه فعال‌ها
          );
        } else {
          this.paymentGatewayList = [...activeGateways];
        }

        const getDefaultFromActivations = (): number | null => {
          if (this.checkoutActivations?.isDefaultJibit) return 452;
          if (this.checkoutActivations?.isDefaultVandar) return 453;
          if (this.checkoutActivations?.isDefaultShepa) return 454;
          if (this.checkoutActivations?.isDefaultZarinpal) return 455;
          if (this.checkoutActivations?.isDefaultZarinLink) return 457;
          if (this.checkoutActivations?.isDefaultSepal) return 458;
          if (this.checkoutActivations?.isDefaultMellat) return 451;
          return null;
        };

        const defaultFromActivations = getDefaultFromActivations();
        let chosen: number | null = null;
        if (
          defaultFromActivations &&
          this.paymentGatewayList.some(
            (pg) => Number(pg.value) === Number(defaultFromActivations)
          )
        ) {
          chosen = Number(defaultFromActivations);
        } else if (
          this.dataScenario &&
          this.dataScenario.onlinePortal &&
          this.paymentGatewayList.some(
            (pg) => Number(pg.value) === Number(this.dataScenario.onlinePortal)
          )
        ) {
          chosen = Number(this.dataScenario.onlinePortal);
        } else if (this.paymentGatewayList.length === 1) {
          chosen = Number(this.paymentGatewayList[0].value);
        } else if (this.paymentGatewayList.length > 0) {
          chosen = Number(this.paymentGatewayList[0].value);
        } else {
          chosen = null;
        }

        if (chosen !== null) {
          this.bankPorts = String(chosen);
        } else {
          this.bankPorts = null;
        }

        this.bankPortsChange.emit(this.bankPorts);
        this.cdr.detectChanges();
      });
  }

  ngOnInit() {
    this.getBankAccounts();
    this.checkForScenario();
    this.cdr.detectChanges();
  }

  ngOnDestroy() {
    this.subscriptions.unsubscribe();
  }

  checkForScenario() {
    if (!this.dataCheckout || !this.allScenarios) return;
    const finalAmount = this.dataCheckout.finalAmount ?? 0;
    const paidAmount = this.dataCheckout.paidAmount ?? 0;
    const isPaid = paidAmount && paidAmount !== 0;
    const targetSortNumber = isPaid ? 2 : 1;
    const validScenarios = this.allScenarios.filter((item: any) => {
      const isSortMatch = item.sortNumber === targetSortNumber;
      const minValid =
        !item.minimumAmountPaidToStart ||
        finalAmount >= +item.minimumAmountPaidToStart;
      const maxValid =
        !item.maximumAmountPaidToFinish ||
        finalAmount < +item.maximumAmountPaidToFinish;
      return isSortMatch && minValid && maxValid;
    });
    const selectedScenario = cloneDeep(validScenarios[0]);
    this.dataScenario = selectedScenario
      ? { ...selectedScenario, isAllowed: true }
      : null;
    setTimeout(() => {
      this.choosePaymentGateWay();
      this.cdr.detectChanges();
    });
  }

  select(event: any) {
    this.bankPorts = event.value;
    this.bankPortsChange.emit(this.bankPorts);
  }

  getBankAccounts() {
    this.isLoading = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(Address.BankAccountGet, {}, false)
        .pipe(
          finalize(() => {
            this.isLoading = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: any) => {
            this.bankAccountList = response.data || [];
            if (this.bankAccountList.length > 0) {
              for (let item of this.bankAccountList) {
                item.fullTitle = item.cardNumber + ' ' + item.bankName;
              }
              const defaultBank = this.bankAccountList.find(
                (b) => b.isDefault === true
              );
              if (defaultBank) {
                this.bankId = String(defaultBank.bankAccountId);
              } else {
                this.bankId = String(this.bankAccountList[0].bankAccountId);
              }
              if (this.bankId) {
                this.bankIdChange.emit(this.bankId);
              }
              this.bankIdChange.emit(this.bankId);
              for (let item of this.bankAccountList) {
                this.getFileUrl(item.avatarId);

                
              }
            }
            this.cdr.detectChanges();
          },
          (err) => {}
        )
    );
  }

  getFileUrl(imageId: string) {
    this.isLoadingGetFile = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .getResponseText(Address.GetFileUrl + imageId)
        .pipe(
          finalize(() => {
            this.isLoadingGetFile = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response) => {
            if (response) {
              for (let item of this.bankAccountList) {
                if (item.avatarId == imageId) {
                  item.fileUrl = response;
                }
              }
            }
            this.cdr.detectChanges();
          },
          (err: any) => {},
          () => {}
        )
    );
  }

  selectBankAccount() {
    this.bankIdChange.emit(this.bankId);
  }

  addBankAccount() {
    const ref = this.dialogService.open(AddBankAccountComponent, {
      header: 'افزودن حساب بانکی',
      dismissableMask: true,
      data: { isForOrdersPayment: true },
      styleClass: 'adding-bank-account-modal',
    });

    ref.onClose.subscribe((model: any) => {
      if (model) {
        this.cdr.markForCheck();
        this._alert.success('حساب بانکی جدید با موفقیت افزوده شد');
        this.getBankAccounts();
      }
    });
  }
}
