import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  EventEmitter,
  Input,
  OnChanges,
  OnInit,
  Output,
} from '@angular/core';
import _, { cloneDeep } from 'lodash';
import { removeItemFromArray } from '@helpers/object.helper';
import { TranslateService } from '@ngx-translate/core';
import { DiscountCodeStore } from '@stores/getDiscount.store';
import { UserStore } from '@stores/user.store';
import { UserProfile } from '@interfaces/user/userProfile';
import { WalletRechargeComponent } from '../../user/wallet/wallet-rechage/wallet-recharge.component';
import { DialogService } from 'primeng/dynamicdialog';
import { stringIsEmptyOrNull } from '@helpers/validator.helper';
import { GetScenarioCheckout } from '@models/orders/getScenarioCheckoutModel';
import { OrderSettingStore } from '@stores/orderSetting.store';
import { MethodGatewayModel } from '@models/orders/MethodGatewayModel';
import { GetSettingCheckout } from '@models/orders/getSetngCheckoutModel';
import { Address } from '@enums/address.enum';
import { finalize, Subscription } from 'rxjs';
import { CommonService } from '@services/common.service';
import { ScenarioStore } from '@stores/scenario.store';

@Component({
  selector: 'paymentMethod',
  templateUrl: 'paymentMethod.component.html',
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class PaymentMethodComponent implements OnInit, OnChanges {
  isLoadingGet: boolean = true;
  @Input() payTypes: string;
  @Input() orderType: string;
  @Input() valueType: string;
  @Input() dataCheckout: any = null;
  @Input() checkoutActivations: any;
  @Output() payTypesChange: EventEmitter<string> = new EventEmitter<string>();
  @Output() isDisableWalletChange: EventEmitter<boolean> =
    new EventEmitter<boolean>();
  @Output() orderSettingValue: EventEmitter<any> = new EventEmitter();
  @Input() bankPorts: MethodGatewayModel;
  @Output() bankPortsChange = new EventEmitter();
  @Input() bankId: string = null;
  @Output() bankIdChange = new EventEmitter();
  afterDiscountPrice: any;
  paymentMethodList = [];
  maximumPrice: number;
  lock: boolean = false;
  dataUser: UserProfile = new UserProfile();
  isDisableWallet: boolean;
  rechargeValue: number = 0;
  url: string;
  selectedScenarioId: number;
  subscriptions = new Subscription();
  validScenarios: any[] = [];
  settingCheckoutModel: GetSettingCheckout = new GetSettingCheckout();
  paymentStages: Array<{
    sortNumber: number;
    title: string;
    methods: any[];
    displayAmount: number;
    minScenarioAmount: number;
    status: 'completed' | 'current' | 'locked';
  }> = [];
  currentPaymentStage: number | null = null;
  selectedScenarioByMethod: { [methodValue: string]: number } = {};
  @Input() checkBankAccount: boolean;
  allScenarios: any[] = [];

  constructor(
    private cdr: ChangeDetectorRef,
    private _translateService: TranslateService,
    public _dicountCodeStore: DiscountCodeStore,
    private _userStore: UserStore,
    public dialogService: DialogService,
    private _orderSettingStore: OrderSettingStore,
    private _commonService: CommonService,
    private _scenarioStore: ScenarioStore,
  ) {
    this.url = globalThis.location?.href;
    this.getStoreDiscountedBeforeAmount();
    this._orderSettingStore.getOrderSettingBehavior().subscribe((res) => {
      if (res != null) {
        this.maximumPrice = res.maximumAmountOnlinePort;
      }
    });
  }

  postIsDiablesWallet() {
    this.isDisableWalletChange.emit(this.isDisableWallet);
  }

  getStoreDiscountedBeforeAmount() {
    this._dicountCodeStore.afterDiscount.subscribe((res) => {
      if (res != null || res != undefined) {
        this.dataCheckout.discountedFinalAmount = res;
      } else {
        this.dataCheckout.discountedFinalAmount = null;
      }
      this.getStorDiscount();
    });
  }

  getStorDiscount() {
    this._dicountCodeStore.discount.subscribe((res) => {
      if (res != null || res != undefined) {
        this.dataCheckout.discount = res;
      } else {
        this.dataCheckout.discount = null;
      }
      this.choosePaymentMethod(this.maximumPrice);
    });
  }

  ngOnInit() {
    this.getUserStore();
    this.orderSetting();
  }

  onGateData(data: any) {
    this.orderSettingValue.emit(this.orderSettingValue);
  }

  getUserStore() {
    this._userStore.getUserBehavior().subscribe((res) => {
      if (res != null) {
        this.dataUser = res;
        this.checkActiveWallet();
      }
    });
  }

  ngOnChanges() {
    if (!this.lock) {
      this.isLoadingGet = false;
      if (
        this.validScenarios.length === 0 &&
        this.dataCheckout != null &&
        !_.isEmpty(this.dataCheckout) &&
        this.checkoutActivations != null &&
        !_.isEmpty(this.checkoutActivations)
      ) {
        this.orderScenarioGetAll();
      }
      this.cdr.detectChanges();
    }
    this.payTypesChange.emit(this.payTypes);
  }

  selectItem(item: any) {
    if (!item || !item.value) return;

    const stage = this.paymentStages.find((s) =>
      s.methods.some((m: any) => Number(m.value) === Number(item.value))
    );
    if (stage) {
      if (stage.status !== 'current') {
        return;
      }
    }

    this.payTypes = String(item.value);
    this.payTypesChange.emit(this.payTypes);
    this.postIsDiablesWallet();
  }

  orderScenarioGetAll() {
    this.subscriptions.add(
      this._commonService
        .put(Address.ScenarioGetAll, {
          orderType: this.dataCheckout.orderType,
        })
        .pipe(finalize(() => {}))
        .subscribe((response) => {
          if (response.data) {
            this.allScenarios = response.data;
            if(this.allScenarios){
              this._scenarioStore.scenariosBehavior.next(this.allScenarios);
            }
            if (this.dataCheckout) {
              const finalAmount = this.dataCheckout.finalAmount || 0;
              const paidAmount = this.dataCheckout.paidAmount || 0;
              const isPaid = paidAmount && paidAmount !== 0;
              const targetSortNumber = isPaid ? 2 : 1;
              this.currentPaymentStage = targetSortNumber;
              const validScenarios = this.allScenarios.filter((item: any) => {
                const minValid =
                  !item.minimumAmountPaidToStart ||
                  finalAmount >= +item.minimumAmountPaidToStart;
                const maxValid =
                  !item.maximumAmountPaidToFinish ||
                  finalAmount < +item.maximumAmountPaidToFinish;
                return minValid && maxValid;
              });
              this.validScenarios = cloneDeep(validScenarios);
              this.fillOutPaymentMethodsFromScenarios();
              this.cdr.detectChanges();
            }
          }
        })
    );
  }

  fillOutPaymentMethodsFromScenarios() {
    this.paymentStages = [];
    this.paymentMethodList = [];
    if (!this.validScenarios || this.validScenarios.length === 0) return;

    this._translateService
      .get('paymentMethod.data')
      .subscribe((paymentMethodData: any[]) => {
        const {
          finalAmountGlobal,
          discountedFinalGlobal,
          finalWageAmountGlobal,
          paidAmountGlobal,
          isDiscount,
        } = this.extractCheckoutData();

        const stagesMap = this.buildStagesMap(
          paymentMethodData,
          finalAmountGlobal,
          discountedFinalGlobal,
          finalWageAmountGlobal,
          isDiscount
        );

        const stageInfos = this.prepareStageInfos(stagesMap);

        this.buildPaymentStages(
          stageInfos,
          finalAmountGlobal,
          discountedFinalGlobal,
          paidAmountGlobal
        );

        this.setDefaultPaymentMethod();

        this.cdr.detectChanges();
      });
  }

  private extractCheckoutData() {
    return {
      finalAmountGlobal: Number(this.dataCheckout.finalAmount) || 0,
      discountedFinalGlobal:
        Number(this.dataCheckout.discountedFinalAmount) || 0,
      finalWageAmountGlobal: Number(this.dataCheckout.finalWageAmount) || 0,
      paidAmountGlobal: Number(this.dataCheckout.paidAmount) || 0,
      isDiscount: !!this.dataCheckout.discount,
    };
  }

  private buildStagesMap(
    paymentMethodData: any[],
    finalAmountGlobal: number,
    discountedFinalGlobal: number,
    finalWageAmountGlobal: number,
    isDiscount: boolean
  ) {
    const stagesMap: {
      [sortNum: number]: {
        scenarios: any[];
        addedValues: number[];
        methods: any[];
      };
    } = {};

    for (const scenario of this.validScenarios) {
      const sortNumber = Number(scenario.sortNumber || 1);
      if (!stagesMap[sortNumber]) {
        stagesMap[sortNumber] = { scenarios: [], addedValues: [], methods: [] };
      }

      const scenarioFinalAmount = this.calculateScenarioFinalAmount(
        scenario,
        finalAmountGlobal,
        discountedFinalGlobal,
        finalWageAmountGlobal,
        isDiscount
      );

      const scenarioWithAmount = { ...scenario, scenarioFinalAmount };
      stagesMap[sortNumber].scenarios.push(scenarioWithAmount);

      this.addPaymentMethod(
        stagesMap[sortNumber],
        scenario,
        paymentMethodData,
        sortNumber
      );
    }

    return stagesMap;
  }

  private calculateScenarioFinalAmount(
    scenario: any,
    finalAmountGlobal: number,
    discountedFinalGlobal: number,
    finalWageAmountGlobal: number,
    isDiscount: boolean
  ): number {
    const payOfOrderWage = !!scenario.paymentOfOrderWage;
    const payWithoutOrderWage = !!scenario.paymentWithoutOrderWage;
    let scenarioFinalAmount = 0;

    if (isDiscount) {
      if (payOfOrderWage) scenarioFinalAmount = finalWageAmountGlobal;
      else if (payWithoutOrderWage)
        scenarioFinalAmount = discountedFinalGlobal - finalWageAmountGlobal;
      else
        scenarioFinalAmount =
          discountedFinalGlobal > 0 ? discountedFinalGlobal : finalAmountGlobal;
    } else if (payOfOrderWage) {
      scenarioFinalAmount = finalWageAmountGlobal;
    } else if (payWithoutOrderWage) {
      scenarioFinalAmount = finalAmountGlobal - finalWageAmountGlobal;
    } else {
      scenarioFinalAmount = finalAmountGlobal;
    }

    return Math.max(Math.floor(Number(scenarioFinalAmount) || 0), 0);
  }

  private addPaymentMethod(
    stage: { addedValues: number[]; methods: any[] },
    scenario: any,
    paymentMethodData: any[],
    sortNumber: number
  ) {
    const methodValue = Number(scenario.paymentMethod);
    if (!methodValue || isNaN(methodValue)) return;

    if (stage.addedValues.includes(methodValue)) return;

    const matchedItem = paymentMethodData.find(
      (item) => Number(item.value) === methodValue
    );

    if (!matchedItem) {
      console.warn(
        `Payment method ${methodValue} not found in translation data`,
        scenario
      );
      return;
    }

    const methodObj = {
      ...matchedItem,
      scenarioId: scenario.scenarioId,
      scenarioName: scenario.paymentMethodName || matchedItem.name,
      sortNumber,
    };

    if (methodValue === 421) {
      stage.methods.unshift(methodObj);
      stage.addedValues.unshift(methodValue);
    } else {
      stage.methods.push(methodObj);
      stage.addedValues.push(methodValue);
    }
  }

  private prepareStageInfos(stagesMap: any) {
    const stageNumbers = Object.keys(stagesMap)
      .map(Number)
      .sort((a, b) => a - b);

    const stageInfos: Array<{
      sortNumber: number;
      minScenarioAmount: number;
      methods: any[];
      title: string;
    }> = [];

    const onlyOneStage = stageNumbers.length === 1;
    let onlyStageIsCombination = false;
    if (onlyOneStage) {
      const onlySn = stageNumbers[0];
      const onlyStageScenarios = stagesMap[onlySn].scenarios || [];
      onlyStageIsCombination = onlyStageScenarios.some(
        (s: any) => !!s.isCombinationOfPaymentMethods
      );
    }

    for (const sn of stageNumbers) {
      const stageInfo = stagesMap[sn];
      const methods = stageInfo.methods;
      if (!stageInfo.scenarios || stageInfo.scenarios.length === 0) continue;
      const amounts = stageInfo.scenarios
        .map((s: any) => Number(s.scenarioFinalAmount) || 0)
        .filter((v: number) => !Number.isNaN(v));

      const minScenarioAmount = amounts.length > 0 ? Math.min(...amounts) : 0;

      let title = '';
      if (onlyOneStage && onlyStageIsCombination) {
        title = 'انتخاب روش پرداخت';
      } else {
        switch (sn) {
          case 1:
            title = 'مرحله اول پرداخت:';
            break;
          case 2:
            title = 'مرحله دوم پرداخت:';
            break;
          default:
            title = `مرحله ${sn} پرداخت:`;
        }
      }

      stageInfos.push({
        sortNumber: sn,
        minScenarioAmount: Math.max(
          Math.floor(Number(minScenarioAmount) || 0),
          0
        ),
        methods,
        title,
      });

      this.paymentMethodList = this.paymentMethodList.concat(methods);
    }

    return stageInfos;
  }

  private buildPaymentStages(
    stageInfos: any[],
    finalAmountGlobal: number,
    discountedFinalGlobal: number,
    paidAmountGlobal: number
  ) {
    let cumulative = 0;

    for (const info of stageInfos) {
      const requiredForThis = info.minScenarioAmount;
      const cumulativeUpToThis = cumulative + requiredForThis;

      let status: 'completed' | 'current' | 'locked';

      if (paidAmountGlobal >= cumulativeUpToThis) {
        status = 'completed';
      } else if (paidAmountGlobal >= cumulative) {
        status = 'current';
      } else {
        status = 'locked';
      }

      const displayAmount = this.calculateDisplayAmount(
        info,
        finalAmountGlobal,
        discountedFinalGlobal,
        paidAmountGlobal
      );

      this.paymentStages.push({
        sortNumber: info.sortNumber,
        title: info.title,
        methods: info.methods,
        displayAmount,
        minScenarioAmount: info.minScenarioAmount,
        status,
      });

      cumulative = cumulativeUpToThis;
    }
  }

  private calculateDisplayAmount(
    info: any,
    finalAmountGlobal: number,
    discountedFinalGlobal: number,
    paidAmountGlobal: number
  ): number {
    const remainingTotal = Math.max(finalAmountGlobal - paidAmountGlobal, 0);
    let displayAmount = 0;

    if (info.sortNumber === 1) {
      if (info.minScenarioAmount > 0) {
        displayAmount = info.minScenarioAmount;
      } else {
        displayAmount =
          discountedFinalGlobal > 0 ? discountedFinalGlobal : finalAmountGlobal;
      }
    } else if (info.minScenarioAmount > 0) {
      displayAmount = Math.min(info.minScenarioAmount, remainingTotal);
    } else {
      displayAmount = remainingTotal;
    }

    return Math.max(Math.floor(Number(displayAmount) || 0), 0);
  }

  private setDefaultPaymentMethod() {
    let defaultMethod = null;
    const currentStageObj = this.paymentStages.find(
      (s) => s.status === 'current'
    );

    if (currentStageObj?.methods?.length) {
      defaultMethod = currentStageObj.methods[0];
    } else if (this.paymentMethodList?.length) {
      defaultMethod = this.paymentMethodList[0];
    }

    if (defaultMethod) {
      this.payTypes = String(defaultMethod.value);
      this.payTypesChange.emit(this.payTypes);
    }
  }

  choosePaymentMethod(maximum: number) {
    if (
      this.dataCheckout &&
      this.paymentMethodList &&
      this.paymentMethodList.length > 0
    ) {
      this.lock = true;
    }
    this.fillOutPaymentMethodsFromScenarios();

    setTimeout(() => {
      if (
        stringIsEmptyOrNull(this.checkoutActivations.value) ||
        this.checkoutActivations.value == 'Buye'
      ) {
        if (
          ((this.dataCheckout && !this.dataCheckout.discount) ||
            (this.dataCheckout && this.dataCheckout.discount)) &&
          this.paymentMethodList &&
          this.paymentMethodList.length > 0
        ) {
          this.payTypes = '421';
          this.paymentMethodList = removeItemFromArray(
            this.paymentMethodList,
            '423',
            'value'
          );
          this.paymentMethodList = removeItemFromArray(
            this.paymentMethodList,
            '425',
            'value'
          );
        }
      } else if (
        this.dataCheckout &&
        this.paymentMethodList &&
        this.paymentMethodList.length > 0
      ) {
        if (this.paymentMethodList.length > 1) {
          this.payTypes = '428';
        } else {
          this.payTypes = '429';
        }
      }
      this.checkActiveWallet();
    }, 1);
  }

  checkActiveWallet() {
    if (
      this.dataUser != undefined &&
      this.dataCheckout != undefined &&
      this.dataUser != null &&
      this.dataCheckout != null &&
      this.dataUser.amountWallet < this.dataCheckout.finalAmount
    ) {
      this.isDisableWallet = true;
      if (this.dataCheckout.discountedFinalAmount) {
        this.rechargeValue =
          this.dataCheckout.discountedFinalAmount - this.dataUser.amountWallet;
      } else {
        this.rechargeValue =
          this.dataCheckout.finalAmount - this.dataUser.amountWallet;
      }
    }

    if (this.paymentMethodList && this.paymentMethodList.length > 0) {
      this.payTypes = this.paymentMethodList[0].value;
      this.payTypesChange.emit(this.payTypes);
    }

    this.cdr.detectChanges();
  }

  // checkActiveWallet() {
  //   if (!this.dataUser || !this.dataCheckout) {
  //     this.isDisableWallet = false;
  //     this.rechargeValue = 0;
  //     return;
  //   }

  //   const walletBalance = Number(this.dataUser.amountWallet) || 0;
  //   const currentStage =
  //     (this.paymentStages &&
  //       this.paymentStages.find((s) => s.status === 'current')) ||
  //     (this.paymentStages && this.paymentStages[0]) ||
  //     null;

  //   if (!currentStage) {
  //     const finalAmt = Number(this.dataCheckout.finalAmount) || 0;
  //     if (walletBalance < finalAmt) {
  //       this.isDisableWallet = true;
  //       const discounted = Number(this.dataCheckout.discountedFinalAmount) || 0;
  //       this.rechargeValue =
  //         (discounted > 0 ? discounted : finalAmt) - walletBalance;
  //     } else {
  //       this.isDisableWallet = false;
  //       this.rechargeValue = 0;
  //     }

  //     if (this.paymentMethodList && this.paymentMethodList.length > 0) {
  //       this.payTypes = this.paymentMethodList[0].value;
  //       this.payTypesChange.emit(this.payTypes);
  //     }

  //     this.cdr.detectChanges();
  //     return;
  //   }

  //   const WALLET_METHOD_VALUE = 426;
  //   const walletAvailableInCurrent = !!(
  //     currentStage.methods &&
  //     currentStage.methods.some(
  //       (m: any) => Number(m.value) === WALLET_METHOD_VALUE
  //     )
  //   );

  //   if (!walletAvailableInCurrent) {
  //     this.isDisableWallet = false;
  //     this.rechargeValue = 0;
  //   } else {
  //     const requiredForStage = Number(currentStage.displayAmount) || 0;

  //     if (walletBalance < requiredForStage) {
  //       this.isDisableWallet = true;
  //       this.rechargeValue = requiredForStage - walletBalance;
  //     } else {
  //       this.isDisableWallet = false;
  //       this.rechargeValue = 0;
  //     }
  //   }

  //   if (this.paymentMethodList && this.paymentMethodList.length > 0) {
  //     this.payTypes = this.paymentMethodList[0].value;
  //     this.payTypesChange.emit(this.payTypes);
  //   }

  //   this.cdr.detectChanges();
  // }

  walletRecharge(item: any = null) {
    const ref = this.dialogService.open(WalletRechargeComponent, {
      header:
        this._translateService.translations[this._translateService.currentLang][
          'walletRecharge'
        ],
      height: '70%',
      width: '40%',
      styleClass: 'wallet-modal',
      data: {
        value: this.rechargeValue,
        url: this.url,
      },
    });
    ref.onClose.subscribe((model: any) => {});
  }

  orderSetting() {
    this.subscriptions.add(
      this._commonService
        .put(Address.OrderSettingGet, {
          orderType: this.orderType,
          valueType: this.valueType,
          isArchive: false,
        })
        .pipe(finalize(() => {}))
        .subscribe((response) => {
          if (response.data) {
            this.checkBankAccount = response.data.checkBankAccount;
            this._orderSettingStore.orderSettingBehavior.next({
              maximumAmountOnlinePort: response.data.maximumAmountOnlinePort,
              minimumAmountOnlinePort: response.data.minimumAmountOnlinePort,
            });
            this.orderSettingValue.emit(response.data);
          }
        })
    );
  }

  get shouldShowBankTransferAlert(): boolean {
    if (!this.allScenarios?.length) return false;
    const hasSecondStageBankTransfer = this.allScenarios.some(
      (s: any) => s.sortNumber === 2 && s.paymentMethod === 431
    );
    return hasSecondStageBankTransfer;
  }
}
