import { Router } from '@angular/router';
import {
  Component,
  OnDestroy,
  OnInit,
  ChangeDetectionStrategy,
  ChangeDetectorRef,
} from '@angular/core';
import { Address } from '@enums/address.enum';
import { AlertHelper } from '@helpers/alert.helper';
import { CommonService } from '@services/common.service';
import { finalize, Subscription } from 'rxjs';
import { response } from '@interfaces/response';
import { ConfirmationService, Message } from 'primeng/api';
import { DialogService } from 'primeng/dynamicdialog';
import { PagingHelper } from '@helpers/paging.helper';
import { MessageStatic } from '@enums/messageStatic.enum';
import { convertDateToGeorgian } from '@helpers/dateTime.helper';
import { OrderListModel } from '@models/orders/OrderListModel';
import { cloneDeep } from 'lodash-es';
import { ColumnKey } from '@enums/columnKey.enum';
import { OrderDetailsComponent } from './components/orderDetails/orderDetails.component';
import { Icon } from '@enums/iconKey.enum';
import { StaticModel } from '@models/dbo/StaticModel';
import { BaseParams } from 'src/app/core/params/user/baseParams';
import { OrderSearchModel } from 'src/app/core/params/order/OrderSearchModel';
import { fadeAnimation } from 'src/app/core/animations/animations';
import { DeviceDetectorService } from 'ngx-device-detector';
import { ValueTypeStore } from '@stores/getValueType.store';
import { stringIsEmptyOrNull } from '@helpers/validator.helper';
import { SurveyModel } from '@models/orders/surveyModel';
import { SurveyComponent } from '../../survey/survey.component';
import { RecieptComponent } from './components/reciept/reciept.component';

@Component({
  selector: 'app-order-list',
  templateUrl: './orderList.component.html',
  changeDetection: ChangeDetectionStrategy.OnPush,
  animations: [fadeAnimation],
  host: { '[@fadeInAnimation]': '' },
})
export class OrderListComponent implements OnInit, OnDestroy {
  constructor(
    private _alert: AlertHelper,
    private _commonService: CommonService,
    private cdr: ChangeDetectorRef,
    private _router: Router,
    private confirmationService: ConfirmationService,
    public dialogService: DialogService,
    public _pagingHelper: PagingHelper,
    public _deviceDetectorService: DeviceDetectorService,
    public _valueTypeStore: ValueTypeStore
  ) {}
  isFromOrderDetails: boolean = false;
  overlayVisible: boolean = false;
  overlayDocuments: boolean = false;
  isLoadingGetDocument: boolean = false;
  isLoadingGetFish: boolean;
  subscriptions = new Subscription();
  messages: Message[] = [];
  data: OrderListModel[] = [];
  isTablet: boolean;
  ticket: any;
  filter: any = {};
  order: any = {};
  isLoading: boolean = false;
  query: BaseParams;
  totalRecords: number = 0;
  search: OrderSearchModel = new OrderSearchModel();
  first: number = 0;
  last: number = 0;
  isShowOrderDetails: boolean;
  message: string = MessageStatic.NotFoundOrder;
  timeout = null;
  orderId: string;
  rowsPerPage = this._pagingHelper.rowsPersPage();
  surveyData: SurveyModel = new SurveyModel();
  statusList: StaticModel[] = [
    { staticId: 181, title: 'پیش فاکتور' },
    { staticId: 183, title: 'در حال بررسی سند واریزی' },
    { staticId: 186, title: 'انجام شده' },
    { staticId: 188, title: 'در انتظار ارسال فیش پرداختی' },
    { staticId: 189, title: 'تایید نشده' },
    { staticId: 190, title: 'در انتظار عودت وجه' },
    { staticId: 191, title: 'در حال بررسی' },
    { staticId: 192, title: 'در انتظار پرداخت وجه باقی مانده' },
    { staticId: 193, title: 'در انتظار اقدام کارشناسان' },
    { staticId: 194, title: 'در انتظار تسویه حساب' },
    { staticId: 195, title: 'تسویه شده' },
  ];
  staticValueType = [];
  isLoadingStatic: boolean = false;
  fileUrl: string;
  isLoadingReciept: boolean;
  fishList: any[] = [];
  fishUrls: any[] = [];
  chipQuery: any;
  overlayChipResult: boolean = false;
  modalSurvey(isFromOrderDetails: boolean = null, item: any = null) {
    if (!stringIsEmptyOrNull(this.surveyData.comment)) {
      this.surveyData.comment = '';
    }
    if (this.surveyData.score) {
      this.surveyData.score = 0;
    }

    const ref = this.dialogService.open(SurveyComponent, {
      data: {
        surveyData: this.surveyData,
        isFromOrderDetails: isFromOrderDetails,
      },
      header: 'نظرسنجی',
      dismissableMask: true,
      styleClass: 'modal-order-details-survey',
      closable: true,
      height: '95%',
      width: '40%',
    });
    ref.onClose.subscribe((model: any) => {
      if (model) {
        this.getData();
      }
      if (item) {
        this.modalDetail(item, model);
      }
    });
  }

  modalDetail(item: any, model: boolean = null) {
    this._valueTypeStore.valueType.next(item.valueType);

    const ref = this.dialogService.open(OrderDetailsComponent, {
      data: {
        id: item ? item.orderId : null,
        orderType: item ? item.orderType : null,
        valueType: item ? item.valueType : null,
        orderNumber: item ? item.orderNumber : null,
        orderTypeName: item ? item.orderTypeName : null,
      },
      header:
        ' سفارش شماره' +
        ' ' +
        item.orderNumber +
        '# ' +
        '- ' +
        item.orderTypeName,
      height: '98%',

      styleClass: 'modal-order-details',
      dismissableMask: true,
    });
    ref.onClose.subscribe((model: any) => {
      if (model === true) {
        this.getData();
      }
    });
  }

  recieptModal(item: any, model: boolean = null) {
    const ref = this.dialogService.open(RecieptComponent, {
      data: {
        id: item ? item.orderId : null,
        documentType: null,
      },
      header:
        ' رسیدهای ارسالی و دریافتی سفارش شماره' +
        ' ' +
        item.orderNumber +
        '# ' +
        '- ' +
        item.orderTypeName,
      height: '98%',
      styleClass: 'modal-reciept',
      dismissableMask: true,
    });
    ref.onClose.subscribe((model: any) => {
      if (model === true) {
        this.getData();
      }
    });
  }

  ngOnInit(): void {

    this.isTablet = this._deviceDetectorService.isTablet();
    this.search.status = -1;
    this.query = new BaseParams();
    Object.assign(this.query, this.search);
  }

  toggle(item: any) {
    item.overlayVisible = !item.overlayVisible;
  }

  open(item: any = null) {
    this.modalDetail(item);
  }

  getDatabyChipSelect(statusId: any) {
    this.search.status = statusId;
    this.getData();
  }

  private _highlightTimeouts: any[] = [];

  clearHighlightTimeouts() {
    if (this._highlightTimeouts && this._highlightTimeouts.length) {
      for (const t of this._highlightTimeouts) {
        clearTimeout(t);
      }
      this._highlightTimeouts = [];
    }
  }

  getData(event: any = null): void {
    this.isLoading = true;
    this.cdr.detectChanges();

    this.prepareQuery(event);
    this.clearHighlightTimeouts();

    this.subscriptions.add(
      this._commonService
        .put(Address.OrderGetAll, this.query)
        .pipe(finalize(() => this.onDataLoadFinalize()))
        .subscribe((res) => this.handleResponse(res))
    );
  }

  private prepareQuery(event: any) {
    this.query = new BaseParams();
    this.search.fromDate = convertDateToGeorgian(this.search.fromDateF);
    this.search.toDate = convertDateToGeorgian(this.search.toDateF);
    if (this.search.status) this.search.status = Number(this.search.status);
    Object.assign(this.query, this.search);
    this.query = this._pagingHelper.requestParamsSet(event, this.query);
  }

  private onDataLoadFinalize() {
    this.isLoading = false;
    this.cdr.detectChanges();
  }

  private handleResponse(res: any) {
    if (!res || !res.data) return;

    this.staticValueType = res.data.valueType;
    const now = Date.now();

    this.data = res.data.map((order) => this.processOrder(order, now));

    this.totalRecords = res[ColumnKey.TotalRecordKey] ?? 0;
    this.first = this._pagingHelper.firstRecord(this.query);
    this.last = this._pagingHelper.lastRecord(this.query, this.totalRecords);

    this.cdr.detectChanges();
  }

  private processOrder(order: any, now: number) {
    const saveDate = new Date(order.saveDate).getTime();
    const elapsed = now - saveDate;
    const fiveMinutesMs = 5 * 60 * 1000;
    const remaining = Math.max(0, fiveMinutesMs - elapsed);
    const isNew = elapsed <= fiveMinutesMs;

    if (isNew && remaining > 0) {
      const t = setTimeout(() => this.clearOrderHighlight(order.id), remaining);
      if (!this._highlightTimeouts) this._highlightTimeouts = [];
      this._highlightTimeouts.push(t);
    }

    return { ...order, isNew };
  }

  private clearOrderHighlight(orderId: any) {
    const idx = this.data.findIndex((d) => d.orderId === orderId);
    if (idx !== -1) {
      this.data[idx].isNew = false;
      this.cdr.detectChanges();
    }
  }

  ngOnDestroy(): void {
    this.clearHighlightTimeouts();
    this.subscriptions.unsubscribe();
  }

  getFileId(item: any) {
    this.isLoadingGetFish = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(Address.GetOrdersDocument, { orderId: item.orderId })
        .pipe(
          finalize(() => {
            this.isLoadingGetFish = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe((res: response) => {
          if (res && res.data) {
            item.fishList = cloneDeep(res.data);
            this.iterateOnDocuments(item);
            this.cdr.detectChanges();
          }
        })
    );
  }

  iterateOnDocuments(item: any) {
    for (let object of item.fishList) {
      this.getDocument(object);
    }
  }

  getDocument(object: any) {
    object.fishUrls = [];
    if (object.imageId && !stringIsEmptyOrNull(object.imageId)) {
      this.isLoadingGetDocument = true;
      this.cdr.detectChanges();
      this.subscriptions.add(
        this._commonService
          .getResponseText(Address.GetFileUrl + object.imageId)
          .pipe(
            finalize(() => {
              this.isLoadingGetDocument = false;
              this.cdr.detectChanges();
            })
          )
          .subscribe((res: response) => {
            object.fishUrls.push(res);

            this.cdr.detectChanges();
          })
      );
    }
  }

  transactionsNavigator(item: OrderListModel) {
    this._router.navigateByUrl(
      '/dashboard/orders/transactions/' + item.orderId
    );
  }

  pay(item: any) {
    if (item.valueType) {
      switch (item.valueType) {
        case 'GFC':
          this._router.navigateByUrl(
            `/dashboard/checkouts/giftcard/${item.orderId}`
          );
          break;
        case 'IWP':
          this._router.navigateByUrl(
            `/dashboard/checkouts/internationalpayment/${item.orderId}`
          );
          break;
        case 'PRM':
          this._router.navigateByUrl(
            `/dashboard/checkouts/premium-accounts/${item.orderId}`
          );
          break;
        case 'FRE':
          this._router.navigateByUrl(
            `/dashboard/checkouts/freeorder/${item.orderId}`
          );
          break;
        case 'ICX':
          this._router.navigateByUrl(
            `/dashboard/checkouts/incomecash/${item.orderId}`
          );
          break;
      }
    }

    // else block removed, directly handle status
    if (!item.valueType && (item.status == 181 || item.status == 192)) {
      switch (item.orderType) {
        case 161:
          this._router.navigateByUrl(
            `/dashboard/checkouts/swift/${item.orderId}`
          );
          break;
        case 909:
          this._router.navigateByUrl(
            `/dashboard/checkouts/paypal/${item.orderId}`
          );
          break;
        case 910:
          this._router.navigateByUrl(
            `/dashboard/checkouts/paypal-recharge/${item.orderId}`
          );
          break;
        case 171:
        case 940:
          this._router.navigateByUrl(
            `/dashboard/checkouts/paypal-sell/${item.orderId}`
          );
          break;
        case 162:
          this._router.navigateByUrl(
            `/dashboard/checkouts/fee/${item.orderId}`
          );
          break;
        case 905:
          this._router.navigateByUrl(
            `/dashboard/checkouts/fee-multiple/${item.orderId}`
          );
          break;
        case 175:
          this._router.navigateByUrl(
            `/dashboard/checkouts/university-tuition/${item.orderId}`
          );
          break;
        case 177:
          this._router.navigateByUrl(
            `/dashboard/checkouts/dormitory-fee/${item.orderId}`
          );
          break;
        case 163:
          this._router.navigateByUrl(
            `/dashboard/checkouts/perfectmoneybuying/${item.orderId}`
          );
          break;
        case 178:
        case 179:
        case 900:
        case 901:
          this._router.navigateByUrl(
            `/dashboard/checkouts/virtualServers/${item.orderId}`
          );
          break;
        case 164:
          this._router.navigateByUrl(
            `/dashboard/checkouts/voucherperfectmoneybuy/${item.orderId}`
          );
          break;
        case 166:
          this._router.navigateByUrl(
            `/dashboard/checkouts/voucherperfectmoneysell/${item.orderId}`
          );
          break;
        case 165:
          this._router.navigateByUrl(
            `/dashboard/checkouts/sellingperfectmoney/${item.orderId}`
          );
          break;
        case 167:
          this._router.navigateByUrl(
            `/dashboard/checkouts/remittancetoIran/${item.orderId}`
          );
          break;
        case 173:
          this._router.navigateByUrl(
            `/dashboard/checkouts/wiseaccountcharge/${item.orderId}`
          );
          break;
        case 907:
          this._router.navigateByUrl(
            `/dashboard/checkouts/sellingwise/${item.orderId}`
          );
          break;
        case 169:
          this._router.navigateByUrl(
            `/dashboard/checkouts/payeeraccountrecharge/${item.orderId}`
          );
          break;
        case 906:
          this._router.navigateByUrl(
            `/dashboard/checkouts/sellingpayeer/${item.orderId}`
          );
          break;
        case 959:
          this._router.navigateByUrl(
            `/dashboard/checkouts/incomecash/${item.orderId}`
          );
          break;
        case 999:
          this._router.navigateByUrl(
            `/dashboard/checkouts/freeorder/${item.orderId}`
          );
          break;
      }
    }

    if (item.status == 188) {
      this._router.navigateByUrl(
        '/dashboard/orders/depositDocument/' +
          item.orderType +
          '/' +
          item.orderId
      );
    }
    if (item.orderType == 907) {
      this._router.navigateByUrl(
        '/dashboard/orders/depositSell/' + item.orderType + '/' + item.orderId
      );
    }
  }

  downloadFish(url) {
    window.open(url, '_blank');
  }

  downloadReciept(item: any) {
    this.isLoadingReciept = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .getResponseText(Address.GetFileUrl + item.fileOrderId)
        .pipe(
          finalize(() => {
            this.isLoadingReciept = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response) => {
            this.fileUrl = response;
            window.open(this.fileUrl, '_blank');

            this.cdr.detectChanges();
          },
          (err: any) => {},
          () => {}
        )
    );
  }

  cancel(item: any) {
    this.confirmationService.confirm({
      message: 'آیا از لغو کردن سفارش مطمئن هستید؟',
      header: 'کنسل کردن سفارش  ' + item.orderNumber + '#',
      icon: Icon.Warning,

      accept: () => {
        item.isLoading = true;
        this.subscriptions.add(
          this._commonService
            .put(Address.OrderCancel, { orderId: item.orderId })
            .pipe(
              finalize(() => {
                item.isLoading = false;
                this.cdr.detectChanges();
              })
            )
            .subscribe(
              (res) => {
                this._alert.success(
                  'سفارش شما با موفقیت لغو گردید ، در صورتی که مبلغی پرداخت کرده اید ظرف 72 ساعت کاری آینده به حساب شما برگشت داده خواهد شد.'
                );
                this.getData();
                this.cdr.detectChanges();
              },
              (error: any) => {},
              () => {}
            )
        );
      },
      reject: () => {},
      key: 'confirmRemove',
    });
  }
}
