import { isGuid } from '@helpers/validator.helper';
import { Router, ActivatedRoute } from '@angular/router';
import {
  Component,
  OnDestroy,
  OnInit,
  ChangeDetectionStrategy,
  ChangeDetectorRef,
} from '@angular/core';
import { Address } from '@enums/address.enum';
import { CommonService } from '@services/common.service';
import { finalize, Subscription } from 'rxjs';
import { OrderModel } from '@models/orders/OrderModel';
import { response } from '@interfaces/response';
import { FileUploadService } from '@services/fileUpload.service';
import { ICurrency } from '@interfaces/blockchain/iCurrency';
import { INetwork } from '@interfaces/blockchain/iNetwork';
import { cloneDeep } from 'lodash-es';

@Component({
  selector: 'payment-with-crypto-currency',
  templateUrl: './paymentWithCryptoCurrency.component.html',
  styleUrls: ['./paymentWithCryptoCurrency.component.scss'],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class PaymentWithCryptoCurrencyComponent implements OnInit, OnDestroy {
  subscriptions = new Subscription();

  constructor(
    private _commonService: CommonService,
    private cdr: ChangeDetectorRef,
    private _router: Router,
    private _activatedroute: ActivatedRoute,
    private _fileUploadService: FileUploadService
  ) {}
  order: OrderModel;
  currencyList: ICurrency[] = [];
  networkList: INetwork[] = [];
  isLoadingGetOrder: boolean = false;
  isLoadingInquiry: boolean = false;
  isLoadingCurrency: boolean = false;
  isLoadingNetwork: boolean = false;
  isLoading: boolean = false;
  isLoadingFile: boolean = false;
  isLoadingGetFile: boolean = false;
  orderId: string = this._activatedroute.snapshot.params['id'];
  code: string = this._activatedroute.snapshot.queryParams['code'];
  currency: ICurrency;
  amount: number;
  network: INetwork;
  fee: number;
  isExpired: boolean = false;
  ngOnInit(): void {
    if (!isGuid(this.orderId)) this._router.navigateByUrl('/dashboard/orders');
    this.getCurrency();
    this.getOrder();
  }

  ngOnDestroy() {
    this.subscriptions.unsubscribe();
  }

  ngOnChanges() {}

  inquiry() {
    if (this.currency && this.network) {
      this.isLoadingInquiry = true;
      this.cdr.detectChanges();
      this.subscriptions.add(
        this._commonService
          .put(
            Address.GetPriceCryptoByOrder,
            {
              orderId: this.orderId,
              marketCode: this.currency.marketCode,
              networkId: this.network.networkId,
            },
            false
          )
          .pipe(
            finalize(() => {
              this.isLoadingInquiry = false;

              this.cdr.detectChanges();
            })
          )
          .subscribe(
            (response: response) => {
              this.amount = response.data.amount;
              this.fee = response.data.fee;
              this.cdr.detectChanges();
            },
            (error) => {}
          )
      );
    } else {
      return;
    }
  }

  getOrder() {
    this.isLoadingGetOrder = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(Address.OrderGetbyId, { orderId: this.orderId }, false)
        .pipe(
          finalize(() => {
            this.isLoadingGetOrder = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this.order = response.data;

            this.cdr.detectChanges();
          },
          (error) => {}
        )
    );
  }

  getCurrency() {
    this.isLoadingCurrency = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService.put(Address.GetCurrencyCryptoByCustomer, {}, false).pipe(
          finalize(() => {
            this.isLoadingCurrency = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this.currencyList = cloneDeep(response.data);
            this._fileUploadService.fileUrlGeneratorPutCurrencyCrypto(this.currencyList,'imageId');
            this.cdr.detectChanges();
          },
          (error) => {}
        )
    );
  }

  getNetwork() {
    if(!this.currency)
    {
      return;
    }
    this.isLoadingNetwork = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(Address.GetNetworkCrypto, {marketCode:this.currency.marketCode}, false)
        .pipe(
          finalize(() => {
            this.isLoadingNetwork = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this.networkList = response.data.filter((item) => item.isActive);
            this.cdr.detectChanges();
          },
          (error) => {}
        )
    );
  }
 
  submit() {
    this.isLoading = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .post( 
          Address.SaveTransactionCryptoCurrency,
          {
            networkId: this.network.networkId,
            marketCode: this.currency.marketCode,
            orderId: this.orderId,
            code: this.code,
          },
          false
        )
        .pipe(
          finalize(() => {
            this.isLoading = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this._router.navigateByUrl(
              '/dashboard/orders/wallet-crypto/' +
                this.orderId +
                '?paymentId=' +
                response.data.paymentId
            );
            this.cdr.detectChanges();
          },
          (error) => {}
        )
    );
  }
}
