import { isGuid, stringIsEmptyOrNull } from '@helpers/validator.helper';
import { Static } from '@interfaces/static';
import { Router, ActivatedRoute } from '@angular/router';
import {
  Component,
  OnDestroy,
  OnInit,
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  AfterViewInit,
} from '@angular/core';
import { Address } from '@enums/address.enum';
import { AlertHelper } from '@helpers/alert.helper';
import { CommonService } from '@services/common.service';
import { finalize, Subscription } from 'rxjs';
import { OrderModel } from '@models/orders/OrderModel';
import { response } from '@interfaces/response';
import { HttpEvent, HttpEventType } from '@angular/common/http';
import { AccountNumberModel } from '@models/orders/AccountNumberModel';
import { getDepositOptions } from '@helpers/values.helper';
import { OrderDocumentModel } from 'src/app/core/params/order/OrderDocumentModel';
import { FileUploadService } from '@services/fileUpload.service';
import { BlockStore } from '@stores/blockOrder';
import { ICurrency } from '@interfaces/blockchain/iCurrency';
import { ColumnKey } from '@enums/columnKey.enum';
import { IpaymentCryptoCurrency } from '@interfaces/orders/ipaymentCryptoCurrency';
import { ConfirmationService } from 'primeng/api';
import { timeDiffrential } from '@helpers/dateTime.helper';
import { INetwork } from '@interfaces/blockchain/iNetwork';

@Component({
  selector: 'wallet-crypto-currency',
  templateUrl: './walletCryptoCurrency.component.html',
  styleUrls: ['./walletCryptoCurrency.component.scss'],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class WalletCryptoCurrencyComponent implements OnInit, OnDestroy {
  subscriptions = new Subscription();

  constructor(
    private _alert: AlertHelper,
    private _commonService: CommonService,
    private cdr: ChangeDetectorRef,
    private _router: Router,
    private _activatedroute: ActivatedRoute,
    private _fileUploadService: FileUploadService,
    private confirmationService: ConfirmationService
  ) {}
  notCompleted: boolean = false;
  order: OrderModel;
  currencyList: ICurrency[] = [];
  isLoadingGetOrder: boolean = false;
  isLoadingUpdateHash: boolean = false;
  isLoadingInquiry: boolean = false;
  isLoadingCurrency: boolean = false;
  isLoading: boolean = false;
  isWaiting: boolean = false;
  isLoadingFile: boolean = false;
  isLoadingGetFile: boolean = false;
  orderId: string = this._activatedroute.snapshot.params['id'];
  code: string = this._activatedroute.snapshot.queryParams['code'];
  currency: ICurrency;
  network: INetwork;
  amount: number;
  fee: number;
  isExpired: boolean = false;
  paymentId: string;
  hashId: string;
  payment: IpaymentCryptoCurrency;
  diffDateSecond: number;
  stopCheck: boolean = false;

  ngOnInit(): void {
    if (!isGuid(this.orderId)) this._router.navigateByUrl('/dashboard/orders');
    this.getOrder();
    this._activatedroute.queryParams.subscribe((params) => {
      if (stringIsEmptyOrNull(params['paymentId']))
        this._router.navigateByUrl('/dashboard/orders');
      else {
        this.paymentId = params['paymentId'];
        this.isWaiting = true;
        this.inquiry();
        setInterval(() => {
          if (!this.stopCheck) {
            this.checkPay();
          }
        }, 40000);
        this.cdr.detectChanges();
      }
    });
  }

  ngOnDestroy() {
    this.subscriptions.unsubscribe();
  }

  ngOnChanges() {}
  inquiry() {
    this.isLoadingInquiry = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(
          Address.GetPaymentCryptoCurrency,
          { paymentId: this.paymentId },
          false
        )
        .pipe(
          finalize(() => {
            this.isLoadingInquiry = false;

            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this.payment = response.data;
            this.payment.wholeAmount = this.payment.amount + this.payment.fee
            this.cdr.detectChanges();
          },
          (error) => {}
        )
    );
  }

  getOrder() {
    this.isLoadingGetOrder = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(Address.OrderGetbyId, { orderId: this.orderId }, false)
        .pipe(
          finalize(() => {
            this.isLoadingGetOrder = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this.order = response.data;

            this.cdr.detectChanges();
          },
          (error) => {}
        )
    );
  }

  checkPay(manualSubmit: boolean = false) {
    if (manualSubmit == true) this.isLoading = true;

    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(
          Address.NipotoCheckTransaction,
          {
            currency: this.payment.currency,
            network: this.payment.network,
            wallet: this.payment.address,
            paymentId: this.paymentId,
          },
          false
        )
        .pipe(
          finalize(() => {
            if (manualSubmit == true) this.isLoading = false;

            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: any) => {
            this.diffDateSecond = timeDiffrential(
              this.payment.saveDate,
              response.currentDate
            );

            if (response.data == true) {
              this.isWaiting = false;
              this.stopCheck = true;
              this.confirmationService.confirm({
                message:
                  'واریز مقدار رمز ارز از جانب شما با موفقیت انجام شده است ، سفارش شما به وضعیت پرداخت شده تغییر پیدا کرد',
                header: 'واریز موفق',
                accept: () => {
                  this._router.navigateByUrl('/dashboard/orders');
                },
                key: 'success',
              });
            } else {
              if (manualSubmit) {
                this.confirmationService.confirm({
                  message:
                    'متاسفانه هنوز نتیجه ی واریز رمز ارز از سمت شما دریافت نشده است ، در نظر داشته باشید که ممکن است باتوجه به شرایط شبکه ی رمز ارز ، واریز تا 20 دقیقه هم زمان ببرد ',
                  header: 'نتیجه استعلام',
                  accept: () => {},
                  key: 'warn',
                });
              }
            }
            this.cdr.detectChanges();
          },
          (error) => {}
        )
    );
  }

  updateHashhId() {
    if (stringIsEmptyOrNull(this.hashId)) {
      this._alert.warning('لطفا تاییدیه تراکنش را وارد نمایید');
      return;
    }
    this.isLoadingUpdateHash = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(
          Address.UpdateByPaymentCryptoCurrency,
          {
            hashId: this.hashId,
            paymentId: this.paymentId,
          },
          false
        )
        .pipe(
          finalize(() => {
            this.isLoadingUpdateHash = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this.confirmationService.confirm({
              message:
                'با تشکر از شما ، درخواست شما با موفقیت ثبت شد ، بزودی کارشناسان ما برای تکمیل فرآیند سفارش با شما تماس خواهند گرفت',
              header: 'ثبت موفق',
              accept: () => {
                this._router.navigateByUrl('/dashboard/orders');
              },
              key: 'success',
            });

            this.cdr.detectChanges();
          },
          (error) => {}
        )
    );
  }
}
