import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  EventEmitter,
  OnDestroy,
  OnInit,
  Output,
  ViewChild,
} from '@angular/core';
import { NgForm } from '@angular/forms';

import { Address } from '@enums/address.enum';
import { AlertHelper } from '@helpers/alert.helper';
import { toEnglishDigits } from '@helpers/convertToEnglishDigit.helper';
import { response } from '@interfaces/response';
import { ICity } from '@interfaces/user/iCity';
import { IProvince } from '@interfaces/user/iProvince';
import { IUserAddress } from '@interfaces/user/iUserAddress';
import { UserProfile } from '@interfaces/user/userProfile';
import { TranslateService } from '@ngx-translate/core';
import { CommonService } from '@services/common.service';
import { StyleStore } from '@stores/style.store';
import { ConfirmationService } from 'primeng/api';
import { DynamicDialogRef } from 'primeng/dynamicdialog';
import {
  Subject,
  Subscription,
  catchError,
  debounceTime,
  distinctUntilChanged,
  filter,
  finalize,
  map,
  of,
  switchMap,
} from 'rxjs';
import { UserAddressSave } from 'src/app/core/params/user/userAddressSave';

@Component({
  selector: 'modal-content',
  templateUrl: 'selectAddressModal.component.html',
  styleUrls: ['selectAddressModal.component.scss'],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SelectAddressModalComponent implements OnInit, OnDestroy {
  @Output() onSave = new EventEmitter();
  subscriptions = new Subscription();
  isLoadingGetAddress: boolean = false;
  isLoadingSaveAddress: boolean = false;
  isLoadingGetProvince: boolean = false;
  isLoadingGetCity: boolean = false;
  data: UserProfile;
  dataAddress: IUserAddress[] = [];
  dataAddressByPostalCode: any;
  selectedAddressId: string | null = null;
  dataProvince: IProvince[] = [];
  selectedProvince: IProvince;
  dataCity: ICity[] = [];
  selectedCity: ICity;
  addressModel: UserAddressSave = new UserAddressSave();
  showAddForm: boolean = false;
  isEditing: boolean = false;
  isLoadingProvinces: boolean = false;
  isLoadingCities: boolean = false;
  isSaving: boolean = false;
  isLoadingGetAddressByPostalCode = false;
  filteredCities: any[] = [];
  filteredProvinces: any[] = [];
  usedProvinceIds = new Set<string>();
  usedCityIds = new Set<string>();
  private postalCode$ = new Subject<string>();
  private LS_PROVINCES_KEY = 'usedProvinces_v1';
  private LS_CITIES_KEY = 'usedCities_v1';
  private persistInLocalStorage = true;
  @Output() addressSelected = new EventEmitter<string | number>();
  @Output() close = new EventEmitter<string>();
  @ViewChild('provinceAC') provinceAC: any;
  @ViewChild('cityAC') cityAC: any;
  @ViewChild('addAddressForm') addAddressForm: NgForm;
  constructor(
    private _commonService: CommonService,
    private cdr: ChangeDetectorRef,
    private _styleStore: StyleStore,
    private _translateService: TranslateService,
    private _alert: AlertHelper,
    private confirmationService: ConfirmationService,
    public ref: DynamicDialogRef
  ) {}

  ngOnInit(): void {
    const sub = this.postalCode$
      .pipe(
        map((v) => (v || '').toString().trim()),
        distinctUntilChanged(),
        debounceTime(600),
        filter((val) => /^\d{10}$/.test(val)),
        switchMap((postalCode) => {
          this.isLoadingGetAddressByPostalCode = true;
          this.cdr.markForCheck();
          return this._commonService
            .put(
              Address.GetAddressByPostalCode,
              { postalCode: postalCode },
              false
            )
            .pipe(
              catchError((err) => {
                return of(null);
              }),
              finalize(() => {
                this.isLoadingGetAddressByPostalCode = false;
                this.cdr.markForCheck();
              })
            );
        })
      )
      .subscribe((response: any) => {
        if (
          response &&
          response.data &&
          response.data.addressInfo &&
          response.data.addressInfo.address
        ) {
          this.addressModel.text = response.data.addressInfo.address;
          this.cdr.markForCheck();
        } else {
        }
      });

    this.subscriptions.add(sub);
    this._styleStore.layoutWith.next({
      width: 'w-full',
      isDashFooter: false,
    });
    this.getUserAddress();
  }

  private loadUsedFromStorage() {
    try {
      const provRaw = localStorage.getItem(this.LS_PROVINCES_KEY);
      const cityRaw = localStorage.getItem(this.LS_CITIES_KEY);
      if (provRaw) {
        const arr = JSON.parse(provRaw) as string[];
        arr.forEach((id) => this.usedProvinceIds.add(id));
      }
      if (cityRaw) {
        const arr = JSON.parse(cityRaw) as string[];
        arr.forEach((id) => this.usedCityIds.add(id));
      }
    } catch (e) {}
  }

  onConfirmSelection() {
    if (!this.selectedAddressId && this.dataAddress?.length) {
      this.selectedAddressId = this.dataAddress[0].addressId;
    }

    this.addressSelected.emit(this.selectedAddressId);
    this.close.emit(this.selectedAddressId);
    this.ref.close(this.selectedAddressId);
  }

  onPostalCodeChange(value: string) {
    this.postalCode$.next(value || '');
    const normalized = toEnglishDigits(value);
    this.addressModel.postalCode = normalized.replace(/\D+/g, '');
  }

  ngOnDestroy(): void {
    this.subscriptions.unsubscribe();
  }

  getUserAddress() {
    this.isLoadingGetAddress = true;
    this.cdr.markForCheck();
    this.subscriptions.add(
      this._commonService
        .put(Address.UserAddressGet, {}, false)
        .pipe(
          finalize(() => {
            this.isLoadingGetAddress = false;
            this.cdr.markForCheck();
          })
        )
        .subscribe((response: response) => {
          this.dataAddress = response?.data;
          this.selectedAddressId = this.dataAddress[0]?.addressId;
        })
    );
  }

  getUserAddressFromPostalCode() {
    this.isLoadingGetAddress = true;
    this.cdr.markForCheck();
    this.subscriptions.add(
      this._commonService
        .put(
          Address.GetAddressByPostalCode,
          { postalCode: this.addressModel.postalCode },
          false
        )
        .pipe(
          finalize(() => {
            this.isLoadingGetAddress = false;
            this.cdr.markForCheck();
          })
        )
        .subscribe((response: response) => {
          this.dataAddressByPostalCode = response.data;
        })
    );
  }

  action() {
    this.isSaving = true;
    if (this.isEditing && this.addressModel) {
      this.updateAddress();
    } else if (!this.isEditing) {
      this.submitNewAddress();
    }
    this.cdr.markForCheck();
  }

  editAddress(item: IUserAddress) {
    if (!this.showAddForm) {
      this.addressModel = { ...item };
      this.getProvince(item.locationIdProvince, item.locationIdCity);
      this.showAddForm = true;
      this.isEditing = true;
      this.cdr.markForCheck();
    }
  }

  updateAddress() {
    this.subscriptions.add(
      this._commonService
        .put(Address.UserUpdateAddress, this.addressModel)
        .pipe(finalize(() => {}))
        .subscribe((res) => {
          this._alert.success(
            this._translateService.translations[
              this._translateService.currentLang
            ]['addressUpdatedSuccesfully']
          );
          this.showAddForm = false;
          this.isEditing = false;
          this.isSaving = false;
          this.addressModel = new UserAddressSave();
          this.getUserAddress();
        })
    );
  }

  deleteAddress(param: any) {
    let item = null;
    if (!param) return;
    if (param) item = param;
    const addressId = item?.addressId ?? param;
    this.confirmationService.confirm({
      acceptLabel:
        this._translateService.translations[this._translateService.currentLang][
          'yes'
        ],
      key: 'deleteAddressConfirmDialog',
      rejectLabel:
        this._translateService.translations[this._translateService.currentLang][
          'no'
        ],
      message:
        this._translateService.translations[this._translateService.currentLang][
          'DeleteAddressAlarm'
        ],
      header:
        this._translateService.translations[this._translateService.currentLang][
          'deleteAddress'
        ],
      icon: 'pi pi-exclamation-triangle',
      accept: () => {
        this.subscriptions.add(
          this._commonService
            .delete(Address.UserDeleteAddress, { addressId })
            .pipe(
              finalize(() => {
                this.cdr.markForCheck();
              })
            )
            .subscribe(
              (res: any) => {
                this._alert.success('حذف آدرس با موفقیت انجام شد');
                this.getUserAddress();
              },
              (err) => {}
            )
        );
      },
    });
  }

  getProvince(id: string = null, cityId: string = null) {
    this.isLoadingGetProvince = true;
    this.cdr.markForCheck();
    this.subscriptions.add(
      this._commonService
        .put(Address.GetProvince, {}, false)
        .pipe(
          finalize(() => {
            this.isLoadingGetProvince = false;
            this.cdr.markForCheck();
          })
        )
        .subscribe((response: response) => {
          this.dataProvince = response?.data ?? [];

          if (!id) {
            const provinceId =
              this.selectedProvince?.locationId ??
              (this.dataProvince.length
                ? this.dataProvince[0].locationId
                : null);

            if (provinceId) {
              this.getCity(provinceId);
            } else {
              this.dataCity = [];
            }
            return;
          }
          this.selectedProvince =
            this.dataProvince.find((p: any) => p.locationId === id) ?? null;
          if (id) {
            this.getCity(id, cityId);
          }
        })
    );
  }

  searchProvince(event: { query: string }) {
    const q = (event.query || '').toLowerCase();
    if (!q) {
      this.filteredProvinces = [...this.dataProvince];
    } else {
      this.filteredProvinces = this.dataProvince.filter((p) =>
        p.title.toLowerCase().includes(q)
      );
    }
  }

  getCity(provinceId: string = null, cityId: string = null) {
    if (!this.addressModel)
      this.addressModel = {
        locationId: null,
        title: null,
        text: null,
        postalCode: null,
        provinceId: null,
        addressId: null,
      };

    this.addressModel.locationId = null;
    if (!provinceId) {
      this.dataCity = [];
      return;
    }
    this.isLoadingGetCity = true;
    this.cdr.markForCheck();
    this.subscriptions.add(
      this._commonService
        .put(Address.GetCity, { locationId: provinceId }, false)
        .pipe(
          finalize(() => {
            this.isLoadingGetCity = false;
            this.cdr.markForCheck();
          })
        )
        .subscribe((response: response) => {
          this.dataCity = response.data;
          if (cityId) {
            this.selectedCity = this.dataCity.find(
              (c: any) => c.locationId === cityId
            );
          }
          if (this.persistInLocalStorage) {
            this.loadUsedFromStorage();
          }
        })
    );
  }

  searchCity(event: { query: string }) {
    const q = (event.query || '').toLowerCase();
    if (!q) {
      this.filteredCities = [...this.dataCity];
    } else {
      this.filteredCities = this.dataCity.filter((c) =>
        c.title.toLowerCase().includes(q)
      );
    }
    this.cdr.markForCheck();
  }

  selectAddress(id: string) {
    this.selectedAddressId = id;
    this.cdr.markForCheck();
  }

  addNewAddress() {
    if (!this.showAddForm) {
      this.showAddForm = true;
      this.getProvince();
    }
    this.selectedProvince = null;
    this.selectedCity = null;
    this.filteredProvinces = [];
    this.filteredCities = [];
    try {
      if (
        this.addAddressForm &&
        typeof this.addAddressForm.resetForm === 'function'
      ) {
        this.addAddressForm.resetForm();
      }
    } catch (e) {}
    setTimeout(() => {
      const provInput = document.getElementById(
        'provinceId'
      ) as HTMLInputElement;
      if (provInput) provInput.value = '';
      const cityInput = document.getElementById('cityId') as HTMLInputElement;
      if (cityInput) cityInput.value = '';
      try {
        if (this.provinceAC && this.provinceAC.inputEL) {
          this.provinceAC.inputEL.nativeElement.value = '';
        }
        if (this.cityAC && this.cityAC.inputEL) {
          this.cityAC.inputEL.nativeElement.value = '';
        }
      } catch (err) {}
    }, 0);
  }

  cancelAdd() {
    this.resetAddForm();
    this.showAddForm = false;
    if (this.isEditing) {
      this.isEditing = !this.isEditing;
    }
    this.cdr.markForCheck();
  }

  resetAddForm() {
    this.addressModel = {
      locationId: null,
      title: null,
      postalCode: null,
      text: null,
      provinceId: null,
      addressId: null,
    };
    this.dataProvince = [];
    this.dataCity = [];
    this.cdr.markForCheck();
  }

  submitNewAddress() {
    this.isEditing = false;
    this.addressModel.locationId = this.selectedCity.locationId;
    if (!this.addressModel.locationId) {
      this._alert.error('لطفاً شهر را انتخاب کنید');
      this.isSaving = false;
      return;
    }

    if (!this.addressModel.text) {
      this._alert.error('آدرس را وارد کنید');
      this.isSaving = false;
      return;
    }

    this.cdr.markForCheck();
    this.subscriptions.add(
      this._commonService
        .post(Address.UserAddAddress, this.addressModel, false)
        .pipe(
          finalize(() => {
            this.isSaving = false;
            this.cdr.markForCheck();
          })
        )
        .subscribe(
          (response: any) => {
            this._alert.success(
              this._translateService.translations[
                this._translateService.currentLang
              ]['addressAddedSuccesfully']
            );
            this.getUserAddress();
            this.resetAddForm();
            this.showAddForm = false;
            this.isSaving = false;
          },
          (err: any) => {}
        )
    );
  }
}
