import { ActivatedRoute, NavigationStart, Router } from '@angular/router';
import {
  Component,
  OnDestroy,
  OnInit,
  AfterViewInit,
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  HostListener,
  ViewChild,
  ElementRef,
} from '@angular/core';
import { Address } from '@enums/address.enum';
import { AlertHelper } from '@helpers/alert.helper';
import { UserProfile } from '@interfaces/user/userProfile';
import { CommonService } from '@services/common.service';
import { finalize, Subject, Subscription, takeUntil } from 'rxjs';
import { response } from '@interfaces/response';
import { IncomeCashPriceModel } from '@models/orders/incomecashPriceModel';
import { HttpEvent, HttpEventType } from '@angular/common/http';
import { IncomeCashSaveModel } from 'src/app/core/params/order/incomeCashSaveModel';
import { CurrencyModel } from '@models/orders/CurrencyModel';
import { FileUploadService } from '@services/fileUpload.service';
import { stringIsEmptyOrNull } from '@helpers/validator.helper';
import { DeviceDetectorService } from 'ngx-device-detector';
import { LegalOrderComponent } from 'src/app/modules/services/components/legal-order/legal-order.component';
import { IStatic } from '@interfaces/dbo/iStatic';
import { MenuItem } from 'primeng/api';
import { IExistVerifying } from '@interfaces/user/IExistVerifying';
import { IAdditionalFields } from '@interfaces/orders/iAdditionalFields';
import { IQuestionList } from '@interfaces/dbo/IQuestionList';
import { DialogService } from 'primeng/dynamicdialog';
import { OrderVerifyComponent } from 'src/app/modules/components/modalContentMobile/orderVerify/orderVerify.component';
import { TranslateService } from '@ngx-translate/core';

@Component({
  selector: 'form-IncomeCashOrder',
  templateUrl: './formIncomeCashOrder.component.html',
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class FormIncomeCashOrderComponent implements OnInit, OnDestroy {
  subscriptions = new Subscription();
  isLoadingGet: boolean = false;
  isValidLegalForm: boolean = true;
  isLoadingGetFullOrder: boolean = false;
  isLoadingGetPrice: boolean = false;
  isLoadingGetVerify: boolean = false;
  inquiryIsOk: boolean = false;
  dataCurrency: CurrencyModel[] = [];
  selectedCurrency: CurrencyModel;
  isShowOrder: boolean = true;
  data: UserProfile;
  totalQuestionRecord: number = 0;
  timeout = null;
  fixedEffectiveAmount: number;
  staticData: IStatic;
  isDesktop: boolean;
  dataPrice: IncomeCashPriceModel;
  model: IncomeCashSaveModel = new IncomeCashSaveModel();
  isLoading: boolean = false;
  isLoadingFile: boolean = false;
  isLoadingGetFile: boolean = false;
  isShowFixedEffectiveAmount: boolean = false;
  extensionStore: string;
  canNotUpload: boolean = false;
  displayModal: boolean = false;
  url: string = null;
  file: File;
  progress: number = 0;
  orderType: string;
  keyValues: string;
  isPageLoading: boolean = false;
  notificationBeforeSave: string;
  notificationBeforeSaveType: string;
  dataSteps: MenuItem[] = [];
  additionalFiels: IAdditionalFields[] = [];
  dataQuestion: IQuestionList[] = [];
  dataSetting: IExistVerifying;
  dataStatic: any;
  totalRecord: number = 0;
  imageId: string;
  dataUser: UserProfile;
  errMsg: any;
  isVerifiedInfo: boolean = false;
  private destroyed$ = new Subject<void>();
  private isActive = true;
  private priceRequestSub: Subscription | null = null;
  private timeoutHandle: any = null;

  @ViewChild('amountInput') amountInputRef!: ElementRef;
  @ViewChild(LegalOrderComponent) appChildComponent: LegalOrderComponent;
  @ViewChild('orderVerify') orderVerify: OrderVerifyComponent;
  constructor(
    private _alert: AlertHelper,
    private _commonService: CommonService,
    private cdr: ChangeDetectorRef,
    private _router: Router,
    private _fileUploadService: FileUploadService,
    public _deviceDetectorService: DeviceDetectorService,
    private _activatedRoute: ActivatedRoute,
    public dialogService: DialogService,
    private _translateService: TranslateService
  ) {
    this._router.events.subscribe((event) => {
      if (event instanceof NavigationStart) {
        if (
          event.url.split('/')[4] != undefined &&
          event.url.split('/')[4] != null &&
          event.url.split('/')[4] != ''
        ) {
          const newOrderType = event.url.split('/')[4];
          if (newOrderType && newOrderType !== this.orderType) {
            if (this.dataSetting) {
              this.dataSetting.notificationListBeforeSave = null;
            }
            this.imageId = null;
            this.handleChangeRoute(newOrderType);
          }
        }
      }
    });
  }

  ngOnInit(): void {
    this.clearState();
    this._activatedRoute.params
      .pipe(takeUntil(this.destroyed$))
      .subscribe((params) => {
        this.onRouteParamChanged(params);
      });
    this.isDesktop = this._deviceDetectorService.isDesktop();
    this.orderType = this._activatedRoute.snapshot.params['orderType'];
    this.handleChangeRoute(this.orderType);
    this.cdr.detectChanges();
  }

  private onRouteParamChanged(params: any) {
    if (this.timeoutHandle) {
      clearTimeout(this.timeoutHandle);
      this.timeoutHandle = null;
    }
    if (this.priceRequestSub) {
      this.priceRequestSub.unsubscribe();
      this.priceRequestSub = null;
    }
  }

  ngOnDestroy() {
    this.isActive = false;
    this.destroyed$.next();
    this.destroyed$.complete();

    if (this.timeoutHandle) {
      clearTimeout(this.timeoutHandle);
      this.timeoutHandle = null;
    }
    if (this.priceRequestSub) {
      this.priceRequestSub.unsubscribe();
      this.priceRequestSub = null;
    }
    if (this.subscriptions) {
      this.subscriptions.unsubscribe();
      this.subscriptions = new Subscription();
    }
  }

  getKeyValuesAdditional(keyValue) {
    this.model.additionalValues = keyValue;
  }

  fileChange(event) {
    let fileList: FileList =
      event.target != undefined && event.target != null
        ? event.target.files
        : event;
    if (fileList.length > 0) {
      this.file = fileList[0];
      this.fileUpload();
    }
  }

  fileUpload() {
    this.isLoadingFile = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._fileUploadService
        .fileUpload(this.file.name, this.file, Address.FileUploadBig)
        .subscribe((event: HttpEvent<any>) => {
          switch (event.type) {
            case HttpEventType.Sent:
              break;
            case HttpEventType.ResponseHeader:
              break;
            case HttpEventType.UploadProgress:
              var eventTotal = event.total ? event.total : 0;
              this.progress = Math.round((event.loaded / eventTotal) * 100);
              this.cdr.detectChanges();
              break;
            case HttpEventType.Response:
              if (event.body['data'] != null) {
                this.model.fileId = event.body['data'];
                this.loadFiles(event.body['data']);
                this.isLoadingFile = false;
                this.cdr.detectChanges();
              }
              setTimeout(() => {
                this.progress = 0;
                this.cdr.detectChanges();
              }, 1500);
          }
        })
    );
  }

  loadFiles(id: string) {
    this.isLoadingGetFile = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .getResponseText(Address.GetFileUrl + id)
        .pipe(
          finalize(() => {
            this.isLoadingFile = false;
            this.isLoadingGetFile = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response) => {
            this.url = response;
            this.extensionStore = String(
              this.url.substring(this.url.lastIndexOf('.') + 1, this.url.length)
            ).toLowerCase();
            this.cdr.detectChanges();
          },
          (err: any) => {},
          () => {}
        )
    );
  }

  remove() {
    this.url = null;
    this.cdr.detectChanges();
  }

  private clearState(): void {
    this.model = new IncomeCashSaveModel();
    this.dataPrice = null;
    this.url = null;
    this.file = null;
    this.progress = 0;
    this.extensionStore = null;
    this.inquiryIsOk = false;
    this.dataCurrency = [];
    this.selectedCurrency = null;
    this.dataSteps = [];
    this.staticData = null;
    this.additionalFiels = [];
    this.dataQuestion = [];
    this.dataSetting = null;
    this.dataStatic = null;
    this.totalRecord = 0;
    this.imageId = null;
    this.fixedEffectiveAmount = null;
    this.isShowFixedEffectiveAmount = null;

    if (this.appChildComponent && this.appChildComponent.formAttr) {
      this.appChildComponent.formAttr.reset();
    }

    this.cdr.detectChanges();
  }

  handleChangeRoute(e: string) {
    this.clearState();
    setTimeout(() => {
      this.clearAmountInput();
    });
    this.isPageLoading = true;
    this.orderType = e;
    if (stringIsEmptyOrNull(this.orderType)) {
      this.isPageLoading = false;
      this.cdr.detectChanges();
    } else {
      this.getStaticById();
    }
  }

  private clearAmountInput(): void {
    this.model.amount = null;
    this.fixedEffectiveAmount = null;

    if (this.amountInputRef?.nativeElement) {
      this.amountInputRef.nativeElement.value = '';
    }
  }

  getFullOrder() {
    this.isLoadingGetFullOrder = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(
          Address.OrderGetFullPage,
          {
            isArchive: false,
            orderType: Number(this.orderType),
            sort: 'sortNumber asc',
            pageSize: 50,
            moduleType: 30001,
          },
          false
        )
        .pipe(
          finalize(() => {
            this.isLoadingGetFullOrder = false;
            this.isPageLoading = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            if (response.data.currency && response.data.currency.data)
              this.dataCurrency = response.data.currency.data;

            if (this.dataCurrency && this.dataCurrency.length > 0) {
              this.selectedCurrency = this.dataCurrency[0];
            }

            if (response.data.steps && response.data.steps.data) {
              this.dataSteps = response.data.steps.data;
            }
            if (this.dataCurrency && this.dataCurrency.length == 1) {
              this.selectedCurrency = this.dataCurrency[0];
            }
            if (response.data.setting && response.data.setting.data) {
              this.dataSetting = response.data.setting.data;
            }

            if (
              response.data.question.data &&
              response.data.question.data.length > 0
            ) {
              this.dataQuestion = response.data.question.data;
              this.totalRecord = this.dataQuestion.length;
            }

            if (response.data.static.data) {
              this.dataStatic = response.data.static.data;

              if (this.dataStatic.additionalFields) {
                this.additionalFiels = JSON.parse(
                  this.dataStatic.additionalFields
                );
              }
              if (
                this.dataStatic.imageId &&
                (this.dataStatic.imageId != null ||
                  this.dataStatic.imageId != '')
              ) {
                this.imageId = this.dataStatic.imageId;
              }
            }
            this.cdr.detectChanges();
          },
          (error) => {}
        )
    );
  }

  getPrice() {
    this.model.currencyId = this.selectedCurrency.currencyId;
    this.model.orderType = this.orderType;
    this.subscriptions.add(
      this._commonService
        .put(Address.IncomeCashGetPrice, this.model, false)
        .pipe(
          finalize(() => {
            this.isLoadingGetPrice = false;

            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this.dataPrice = response.data;
            this.inquiryIsOk = true;
            this.cdr.detectChanges();
          },
          (error) => {}
        )
    );
  }

  getStaticById() {
    this.subscriptions.add(
      this._commonService
        .put(Address.GetStaticById, {
          staticId: Number(this.orderType),
        })
        .pipe(
          finalize(() => {
            this.getFullOrder();
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this.staticData = response.data;
            this.fixedEffectiveAmount = this.staticData.fixedEffectiveAmount;

            if (this.staticData.imageId)
              this.loadFilesStaticImage(this.staticData.imageId);
            this.cdr.detectChanges();
          },
          (error) => {}
        )
    );
  }

  loadFilesStaticImage(id: string) {
    this.isLoadingFile = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .getResponseText(Address.GetFileUrl + id)
        .pipe(
          finalize(() => {
            this.isLoadingFile = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response) => {
            this.staticData.fileUrl = response;

            this.cdr.detectChanges();
          },
          (err: any) => {},
          () => {}
        )
    );
  }

  inquiry() {
    if (this.timeoutHandle) {
      clearTimeout(this.timeoutHandle);
      this.timeoutHandle = null;
    }
    const amountRaw = this.model?.amount;
    const amount =
      amountRaw === '' || amountRaw === null || amountRaw === undefined
        ? NaN
        : Number(amountRaw);

    const hasValidAmount = Number.isFinite(amount) && amount >= 0;
    const hasCurrency =
      this.selectedCurrency && this.selectedCurrency.currencyId;

    if (!this.isActive) {
      this.isLoadingGetPrice = false;
      this.dataPrice = null;
      this.cdr.detectChanges();
      return;
    }

    if (hasCurrency && hasValidAmount) {
      this.isLoadingGetPrice = true;
      this.cdr.detectChanges();
      this.timeoutHandle = setTimeout(() => {
        this.timeoutHandle = null;
        this.getPrice();
      }, 900);
    } else {
      this.isLoadingGetPrice = false;
      this.dataPrice = null;
      this.inquiryIsOk = false;
      this.cdr.detectChanges();
    }
  }

  additionalFieldsValid(): boolean {
    if (!this.additionalFiels.length) return true;

    let isValid = true;

    this.additionalFiels.forEach((field) => {
      if (
        !field.isDeleted &&
        (field.persianName?.trim() || field.englishName?.trim()) &&
        field.required === true &&
        (!field.value || String(field.value).trim() === '')
      ) {
        isValid = false;
      }
    });

    return isValid;
  }

  submit() {
    if (!this.additionalFieldsValid()) {
      this._alert.error('لطفا از پر کردن تمام بخش‌های ستاره دار مطمئن شوید.');
      return;
    }
    if (!this.isValidLegalForm) {
      this._alert.warning('لطفا قسمت های اجباری فرم حقوقی را تکمیل نمایید');
      return;
    }
    this.isLoading = true;
    this.model.currencyId = this.selectedCurrency.currencyId;

    this.model.orderType = this.orderType;

    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .post(Address.IncomeCashSave, this.model, false)
        .pipe(
          finalize(() => {
            this.isLoading = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this._alert.success(
              this._translateService.translations[
                this._translateService.currentLang
              ]['checkoutSubmittedSuccesfully']
            );
            this._router.navigateByUrl(
              '/dashboard/checkouts/incomecash/' + response.data.orderId
            );
          },
          (error) => {
            this.errMsg = error.status;
            if ([473, 472, 475, 474, 470, 471, 476].includes(error.status)) {
              setTimeout(() => {
                this.orderVerify.modalGetMobile();
              }, 200);
            }
          }
        )
    );
  }

  isSideFix: boolean = false;
  @HostListener('window:scroll', ['$event'])
  onWindowScroll($event) {
    const numberHeigth = window.scrollY;
    if (numberHeigth >= 90) {
      this.isSideFix = true;
    } else {
      this.isSideFix = false;
    }
  }

  scrollToElement($element): void {
    $element.scrollIntoView({
      behavior: 'smooth',
      block: 'start',
      inline: 'nearest',
    });
  }

  onDialogHide() {
    this.displayModal = false;
  }
}
