import { Component, OnInit, OnDestroy } from '@angular/core';
import { Address } from '@enums/address.enum';
import { UserProfile } from '@interfaces/user/userProfile';
import { CommonService } from '@services/common.service';
import { UserStore } from '@stores/user.store';
import { finalize, Subscription } from 'rxjs';
import { Message } from 'primeng/api';
import { cloneDeep } from 'lodash-es';
import { ColumnKey } from '@enums/columnKey.enum';
import { SignalrService } from '@services/signalr.service';
import { InboxListModel } from '@models/user/InboxListModel';
import { InboxSearch } from '@models/user/InboxSearch';
import { PagingHelper } from '@helpers/paging.helper';
import { BaseParams } from 'src/app/core/params/user/baseParams';
import { TranslateService } from '@ngx-translate/core';
import { DomSanitizer } from '@angular/platform-browser';
import { ActivatedRoute, Router } from '@angular/router';
import { stringIsEmptyOrNull } from '@helpers/validator.helper';
import { fadeAnimation } from 'src/app/core/animations/animations';
import { StyleStore } from '@stores/style.store';

@Component({
  selector: 'inbox',
  templateUrl: 'inbox.component.html',
  styleUrls: ['./inbox.component.scss'],
  animations: [fadeAnimation],
  host: { '[@fadeInAnimation]': '' },
})
export class InboxComponent implements OnInit, OnDestroy {
  index: number = 0;
  data: InboxListModel[] = [];
  isLoadingGet: boolean = false;
  isLoadingGetById: boolean = false;
  isLoading: boolean = false;
  userData: UserProfile = new UserProfile();
  totalRecords: number;
  subscriptions = new Subscription();
  messages: Message[] = [];
  search: InboxSearch = new InboxSearch();
  first: number = 0;
  last: number = 0;
  query: BaseParams = new BaseParams();
  notFoundMessage: string;
  animating: boolean;
  onBeforeToggle: any;
  collapsed: any;
  constructor(
    public signalRService: SignalrService,
    private _userStore: UserStore,
    public _pagingHelper: PagingHelper,
    private _router: Router,
    private _commonService: CommonService,
    private sanitizer: DomSanitizer,
    private translateService: TranslateService,
    private _activatedroute: ActivatedRoute,
    private _styleStore: StyleStore
  ) {}

  ngOnInit(): void {
    this._styleStore.layoutWith.next({
      width: 'w-full',
      isDashFooter: false,
    });
    this.translateService.get('notFound').subscribe((text: string) => {
      this.notFoundMessage = text;
    });

    this._userStore.getUserBehavior().subscribe((res) => {
      if (res != null) {
        this.userData = res;
      }
    });
    this.getData();
  }

  getData(event: any = null): void {
    this.isLoadingGet = true;
    this.query = new BaseParams();
    if (event != null) {
      this.query.startIndex = event.page + 1;
      this.query.pageSize = event.rows;
    }
    
    const currentUrl = this._router.url;
    if (currentUrl.endsWith('/showall')) {
      this.search.isNew = null;
    } else {
      this.search.isNew = true;
    }
    
    Object.assign(this.query, this.search);
    this.query = this._pagingHelper.requestParamsSet(event, this.query);
    this.subscriptions.add(
      this._commonService
        .put(Address.InboxGet, this.query)
        .pipe(
          finalize(() => {
            this.isLoadingGet = false;
          })
        )
        .subscribe((res) => {
          if (res && res.data) {
            this.data = cloneDeep(res.data);
            for (let item of this.data) {
              item.isLoading = true;
              item.isCollapsed = true;
              item.text = this.sanitizer.bypassSecurityTrustHtml(item.text);
              if (
                !stringIsEmptyOrNull(
                  this._activatedroute.snapshot.params['inboxId']
                ) &&
                item.inboxId == this._activatedroute.snapshot.params['inboxId']
              )
                item.isCollapsed = false;
            }
            this.totalRecords = res[ColumnKey.TotalRecordKey]
              ? res[ColumnKey.TotalRecordKey]
              : 0;
            this.first = this._pagingHelper.firstRecord(this.query);
            this.last = this._pagingHelper.lastRecord(
              this.query,
              this.totalRecords
            );
          }
        })
    );
  }

  newEditTicket(item: InboxListModel = null) {
    item == null
      ? this._router.navigateByUrl('/dashboard/tickets/newticket')
      : this._router.navigateByUrl(
          '/dashboard/tickets/showticket/' + item.inboxId
        );
  }

  seenInbox(e: any, item: InboxListModel) {
    if (!e.collapsed) {
      if (!item.isSeen) {
        item.isLoading = true;
        this.subscriptions.add(
          this._commonService
            .put(Address.InboxSeenUpdate, { inboxId: item.inboxId })
            .pipe(
              finalize(() => {
                item.isLoading = false;
              })
            )
            .subscribe((res) => {
              item.isSeen = true;
              this.signalRService
                .sendMessageToHub('InboxGet::' + this.userData.userId)
                .subscribe({});
            })
        );
      } else item.isLoading = false;
    }
  }

  ngOnDestroy() {
    this.subscriptions.unsubscribe();
  }
}
