import { HttpEvent, HttpEventType } from '@angular/common/http';
import {
  ChangeDetectorRef,
  Component,
  OnInit,
  AfterViewInit,
  OnDestroy,
} from '@angular/core';
import { DomSanitizer } from '@angular/platform-browser';
import { ActivatedRoute, Router } from '@angular/router';
import { Address } from '@enums/address.enum';
import { AlertHelper } from '@helpers/alert.helper';
import { isGuid, stringIsEmptyOrNull } from '@helpers/validator.helper';
import { UserProfile } from '@interfaces/user/userProfile';
import { StaticModel } from '@models/dbo/StaticModel';
import { PartListModel } from '@models/user/PartListModel';
import { SubjectListModel } from '@models/user/SubjectListModel';
import { TicketContentList } from '@models/user/TicketContentList';
import { TicketInsertModel } from '@models/user/TicketInsertModel';
import { CommonService } from '@services/common.service';
import { FileUploadService } from '@services/fileUpload.service';
import { UserStore } from '@stores/user.store';
import { cloneDeep } from 'lodash-es';
import { Subscription, finalize } from 'rxjs';
import { fadeAnimation } from 'src/app/core/animations/animations';
import { AttachmentTicketList } from 'src/app/core/params/user/AttachmentTicketList';
import { Location, ViewportScroller } from '@angular/common';
import { IQuestionList } from '@interfaces/dbo/IQuestionList';

@Component({
  selector: 'newTicket',
  templateUrl: 'newTicket.component.html',
  styleUrls: ['newTicket.component.scss'],
  animations: [fadeAnimation],
  host: { '[@fadeInAnimation]': '' },
})
export class NewTicketComponent implements OnInit, AfterViewInit, OnDestroy {
  isLoadingGetSubject: boolean = false;
  isLoadingGetPart: boolean = false;
  subjecData: SubjectListModel[] = [];
  partData: PartListModel[] = [];
  isLoadingStatic: boolean = false;
  subscriptions = new Subscription();
  importanceList: StaticModel[] = [];
  isLoadingGet: boolean = false;
  isLoadingGetContent: boolean = false;
  isLoading: boolean = false;
  isEdit: boolean = false;
  model: TicketInsertModel = new TicketInsertModel();
  userData: UserProfile = new UserProfile();
  modelContent: TicketContentList[] = [];
  ltrEditor: boolean = false;
  fileList: AttachmentTicketList[] = [new AttachmentTicketList()];
  isLoadingGetFull: boolean;
  dataQuestion: IQuestionList[] = [];
  totalRecord: number;
  isLoadingFileUrls: boolean;
  constructor(
    private _activatedroute: ActivatedRoute,
    private cdr: ChangeDetectorRef,
    private _commonService: CommonService,
    private _router: Router,
    private _alert: AlertHelper,
    private sanitizer: DomSanitizer,
    private _userStore: UserStore,
    private _fileUploadService: FileUploadService,
    public location: Location,
    private viewportScroller: ViewportScroller
  ) { }

  ngOnInit() {
    if (!stringIsEmptyOrNull(this._activatedroute.snapshot.params['id'])) {
      this.isEdit = true;
      this.getById();
    } else {
      this.getFullTicket();
    }
    this._userStore.getUserBehavior().subscribe((res) => {
      if (res != null) {
        this.userData = res;
      }
    });
  }

  ngAfterViewInit(): void { }
  setDirection() {
    this.ltrEditor = !this.ltrEditor;
  }

  back() {
    window.history.back();
  }

  goBackToPrevPage() {
    this.location.back();
  }

  getSubject(): void {
    this.isLoadingGetSubject = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(Address.SubjectList, {})
        .pipe(
          finalize(() => {
            this.isLoadingGetSubject = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe((res) => {
          if (res && res.data) {
            this.subjecData = cloneDeep(res.data);
            this.getStaticData();
            this.cdr.detectChanges();
          }
        })
    );
  }

  getById(): void {
    this.isLoadingGet = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(Address.TicketGetById, {
          ticketId: this._activatedroute.snapshot.params['id'],
        })
        .pipe(
          finalize(() => {
            this.isLoadingGet = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe((res) => {
          if (res && res.data) {
            this.model = cloneDeep(res.data);
            this.getAllContent();
            this.cdr.detectChanges();
          }
        })
    );
  }

  getAllContent(): void {
    this.isLoadingGetContent = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(Address.TicketContentGet, {
          ticketId: this._activatedroute.snapshot.params['id'],
        })
        .pipe(
          finalize(() => {
            this.isLoadingGetContent = false;
          })
        )
        .subscribe((res) => {
          if (res && res.data) {
            this.modelContent = cloneDeep(res.data);

            for (let item of this.modelContent) {
              item.text = this.sanitizer.bypassSecurityTrustHtml(item.text);
              if (item.attachments) {
                this._fileUploadService.fileUrlGeneratorPutTicket(item.attachments, 'fileId');
              }
            }
          }
        })
    );
  }

  getPart(): void {
    this.isLoadingGetPart = true;
    this.subscriptions.add(
      this._commonService
        .put(Address.PartList, {})
        .pipe(
          finalize(() => {
            this.isLoadingGetPart = false;
          })
        )
        .subscribe((res) => {
          if (res && res.data) {
            this.partData = cloneDeep(res.data);
          }
        })
    );
  }

  getStaticData(): void {
    this.isLoadingStatic = true;
    this.subscriptions.add(
      this._commonService
        .put(Address.Static, { parentId: 150 })
        .pipe(
          finalize(() => {
            this.isLoadingStatic = false;
          })
        )
        .subscribe((res) => {
          if (res && res.data) {
            this.importanceList = cloneDeep(res.data);
            this.getPart();
          }
        })
    );
  }

  submit() {
    if (!this.isEdit) this.saveTicket();
    else this.saveTicketContent();
  }
  saveTicket() {
    this.isLoading = true;
    this.cdr.detectChanges();
    if (this.fileList != null && this.fileList.length > 0) {
      for (let item of this.fileList) {
        if (item.fileId == null || item.title == null) this.remove(item);
        this.cdr.detectChanges();
      }
    }
    if (this.fileList != null && this.fileList.length > 0) {
      this.model.fileList = JSON.stringify(this.fileList);
    }
    if (this.model.importance)
      this.model.importance = Number(this.model.importance);
    this.subscriptions.add(
      this._commonService
        .post(Address.TicketSave, this.model)
        .pipe(
          finalize(() => {
            this.isLoading = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe((res) => {
          this._alert.success(
            'ارسال تیکت با موفقیت انجام شد ، بزودی مورد بررسی کارشناسان قرار خواهد گرفت'
          );
          this._router.navigateByUrl('/dashboard/tickets');
        })
    );
  }
  saveTicketContent() {
    this.isLoading = true;
    this.cdr.detectChanges();
    if (this.fileList != null && this.fileList.length > 0) {
      for (let item of this.fileList) {
        if (item.fileId == null || item.title == null) this.remove(item);
        this.cdr.detectChanges();
      }
    }
    if (this.fileList != null && this.fileList.length > 0) {
      this.model.fileList = JSON.stringify(this.fileList);
    }
    if (this.model.importance)
      this.model.importance = Number(this.model.importance);
    this.subscriptions.add(
      this._commonService
        .post(Address.TicketContentSave, {
          text: this.model.text,
          ticketId: this._activatedroute.snapshot.params['id'],
          fileList: this.model.fileList,
        })
        .pipe(
          finalize(() => {
            this.isLoading = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe((res) => {
          this._alert.success('ارسال پاسخ با موفقیت انجام شد');
          this._router.navigateByUrl('/dashboard/tickets');
        })
    );
  }
  ngOnDestroy() {
    this.subscriptions.unsubscribe();
  }

  fileChange(e, item: AttachmentTicketList) {
    let fileList: FileList =
      e.target != undefined && e.target != null ? e.target.files : event;
    if (fileList.length > 0) {
      var file = fileList[0];
      this.fileUpload(file, item);
    }
  }

  fileUpload(fileItem: File, data: AttachmentTicketList) {
    data.isLoading = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._fileUploadService
        .fileUpload(fileItem.name, fileItem, Address.FileUploadDocument)
        .subscribe((event: HttpEvent<any>) => {
          switch (event.type) {
            case HttpEventType.Sent:
              break;
            case HttpEventType.ResponseHeader:
              break;
            case HttpEventType.UploadProgress:
              var eventTotal = event.total ? event.total : 0;
              data.progress = Math.round((event.loaded / eventTotal) * 100);
              this.cdr.detectChanges();
              break;
            case HttpEventType.Response:
              if (event.body['data'] != null) {
                data.fileId = event.body['data'];
                data.title = fileItem.name;
                this.loadFiles(event.body['data'], data);
                data.isLoading = false;
                this.cdr.detectChanges();
              }
              setTimeout(() => {
                data.progress = 0;
                this.cdr.detectChanges();
              }, 1500);
          }
        })
    );
  }

  loadFiles(id: string, item: AttachmentTicketList) {
    item.isLoadingGet = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .getResponseText(Address.GetFileUrl + id)
        .pipe(
          finalize(() => {
            item.isLoading = false;
            item.isLoadingGet = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response) => {
            item.url = response;
          },
          (err: any) => { },
          () => { }
        )
    );
  }

  getFullTicket() {
    this.isLoadingGetFull = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(Address.TicketGetFull, { parentId: 150 })
        .pipe(
          finalize(() => {
            this.isLoadingGetFull = false;

            this.cdr.detectChanges();
          })
        )
        .subscribe((response) => {
          if (response.data) {
            if (response.data.level.data) {
              this.importanceList = response.data.level.data;
            }
            if (response.data.part.data) {
              this.partData = response.data.part.data;
            }

            if (response.data.subject.data) {
              this.subjecData = response.data.subject.data;
            }

            if (response.data.question.data) {
              this.dataQuestion = response.data.question.data;
              this.totalRecord = this.dataQuestion.length;
            }
          }
          this.cdr.detectChanges();
        })
    );
  }

  addNewAttach() {
    this.fileList.push(new AttachmentTicketList());
    this.cdr.detectChanges();
  }

  remove(data: AttachmentTicketList) {
    let index = 0;

    this.fileList.splice(index, 1);
    index++;
    this.cdr.detectChanges();
  }

  public anchorScroll(elementId: string): void {
    this.viewportScroller.scrollToAnchor(elementId);
  }

  scrollToElement($element): void {
    $element.scrollIntoView({
      behavior: 'smooth',
      block: 'start',
      inline: 'nearest',
    });
  }
}
