import { ActivatedRoute, Router } from '@angular/router';
import { Component, OnInit, ChangeDetectorRef, OnDestroy } from '@angular/core';
import { Address } from '@enums/address.enum';
import { CommonService } from '@services/common.service';
import { finalize, Subscription } from 'rxjs';
import { Message } from 'primeng/api';
import { DialogService } from 'primeng/dynamicdialog';
import { PagingHelper } from '@helpers/paging.helper';
import { MessageStatic } from '@enums/messageStatic.enum';
import { ColumnKey } from '@enums/columnKey.enum';
import { OrderDetailsComponent } from 'src/app/modules/orders/orderList/components/orderDetails/orderDetails.component';
import { TicketListModel } from '@models/user/TicketListModel';
import { TicketSearch } from '@models/user/TicketSearch';
import { BaseParams } from 'src/app/core/params/user/baseParams';
import { Static } from '@interfaces/static';
import { getTicketStatus } from '@helpers/values.helper';
import { fadeAnimation } from 'src/app/core/animations/animations';
import { StyleStore } from '@stores/style.store';
import { stringIsEmptyOrNull } from '@helpers/validator.helper';

@Component({
  selector: 'tickets',
  templateUrl: 'tickets.component.html',
  animations: [fadeAnimation],
  host: { '[@fadeInAnimation]': '' },
})
export class TicketsComponent implements OnInit, OnDestroy {
  constructor(
    private _commonService: CommonService,
    private cdr: ChangeDetectorRef,
    private _router: Router,
    public dialogService: DialogService,
    public _pagingHelper: PagingHelper,
    private _styleStore: StyleStore,
    private _activatedroute: ActivatedRoute
  ) {}
  subscriptions = new Subscription();
  messages: Message[] = [];
  data: TicketListModel[] = [];
  ticket: any;
  filter: any = {};
  order: any = {};
  isLoading: boolean = false;
  query: BaseParams = new BaseParams();
  totalRecords: number = 0;
  search: TicketSearch = new TicketSearch();
  first: number = 0;
  last: number = 0;
  message: string = MessageStatic.NotFound;
  timeout = null;
  rowsPerPage = this._pagingHelper.rowsPersPage();
  statusList: Static[] = [];
  isLoadingStatic: boolean = false;
  selectedStaticId: number;
  ngOnInit(): void {
    this._styleStore.layoutWith.next({
      width: 'w-full',
      isDashFooter: false,
    });
    this.selectedStaticId = -1;
    Object.assign(this.query, this.search);
    this.statusList = getTicketStatus();
  }

  getDatabyChipSelect(statusId: any) {
    this.selectedStaticId = statusId;
    this.search.status = statusId;
    this.getData();
  }

  newEditTicket(item: TicketListModel = null) {
    if (stringIsEmptyOrNull(this._activatedroute.snapshot.params['language'])) {
      item == null
        ? this._router.navigateByUrl('/dashboard/tickets/newticket')
        : this._router.navigateByUrl(
            '/dashboard/tickets/showticket/' + item.ticketId
          );
    } else {
      item == null
        ? this._router.navigateByUrl(
            '/dashboard/tickets/newticket/' +
              this._activatedroute.snapshot.params['language']
          )
        : this._router.navigateByUrl(
            '/dashboard/tickets/showticket/' +
              item.ticketId +
              '/' +
              this._activatedroute.snapshot.params['language']
          );
    }
  }

  open(item: any = null) {
    const ref = this.dialogService.open(OrderDetailsComponent, {
      data: {
        id: item ? item.orderId : null,
        orderType: item ? item.orderType : null,
      },
      header: 'جزئیات سفارش' + item.orderNumber + '#',
      width: '90%',
      height: '100%',
      dismissableMask: true,
    });
    ref.onClose.subscribe((model: any) => {});
  }

  getData(event: any = null): void {
    this.isLoading = true;
    this.cdr.detectChanges();
    this.query = new BaseParams();
    if (this.search.status) this.search.status = Number(this.search.status);
    Object.assign(this.query, this.search);
    this.query = this._pagingHelper.requestParamsSet(event, this.query);
    this.clearHighlightTimeouts?.();

    this.subscriptions.add(
      this._commonService
        .put(Address.TicketGetAll, this.query)
        .pipe(finalize(() => this.onDataLoadFinalize()))
        .subscribe((res) => this.handleResponse(res))
    );
  }

  private onDataLoadFinalize() {
    this.isLoading = false;
    this.cdr.detectChanges();
  }

  private handleResponse(res: any) {
    if (!res || !res.data) return;

    const now = Date.now();
    this.data = res.data.map((ticket) => this.processTicket(ticket, now));
    this.cdr.detectChanges();

    this.totalRecords = res[ColumnKey.TotalRecordKey] ?? 0;
    this.first = this._pagingHelper.firstRecord(this.query);
    this.last = this._pagingHelper.lastRecord(this.query, this.totalRecords);
  }

  private processTicket(ticket: any, now: number) {
    const saveDate = new Date(ticket.saveDate).getTime();
    const elapsed = now - saveDate;
    const fiveMinutesMs = 5 * 60 * 1000;
    const remaining = Math.max(0, fiveMinutesMs - elapsed);
    const isNew = elapsed <= fiveMinutesMs;
    if (isNew && remaining > 0) {
      const t = setTimeout(
        () => this.clearTicketHighlight(ticket.id),
        remaining
      );
      if (!this._highlightTimeouts) {
        this._highlightTimeouts = [];
      }
      this._highlightTimeouts.push(t);
    }
    return { ...ticket, isNew };
  }

  private clearTicketHighlight(ticketId: any) {
    const idx = this.data.findIndex((d) => d.ticketId === ticketId);
    if (idx !== -1) {
      this.data[idx].isNew = false;
      this.cdr.detectChanges();
    }
  }

  private _highlightTimeouts: any[] = [];

  clearHighlightTimeouts() {
    if (this._highlightTimeouts && this._highlightTimeouts.length) {
      for (const t of this._highlightTimeouts) {
        clearTimeout(t);
      }
      this._highlightTimeouts = [];
    }
  }

  ngOnDestroy(): void {
    this.clearHighlightTimeouts();
    this.subscriptions.unsubscribe();
  }
}
