import {
  AfterViewInit,
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  OnDestroy,
  OnInit,
  ViewChild,
} from '@angular/core';
import { DynamicDialogConfig, DynamicDialogRef } from 'primeng/dynamicdialog';
import { finalize } from 'rxjs/operators';
import { CommonService } from '@services/common.service';
import { Address } from '@enums/address.enum';
import { Subject, Subscription } from 'rxjs';
import { TranslateService } from '@ngx-translate/core';
import { fadeAnimation } from 'src/app/core/animations/animations';
import { setAutoFocous } from '@helpers/window.helper';
import { UserProfile } from '@interfaces/user/userProfile';

@Component({
  selector: 'addBankAccount',
  templateUrl: 'addBankAccount.component.html',
  changeDetection: ChangeDetectionStrategy.OnPush,
  animations: [fadeAnimation],
  host: { '[@fadeInAnimation]': '' },
})
export class AddBankAccountComponent
  implements OnInit, OnDestroy, AfterViewInit
{
  @ViewChild('account') input: ElementRef;
  constructor(
    private _commonService: CommonService,
    public ref: DynamicDialogRef,
    public config: DynamicDialogConfig,
    private cdr: ChangeDetectorRef,
    private _translate: TranslateService
  ) {
    this.dataUser = this.config.data?.dataUser ?? null;
    this.isForOrdersPayment = this.config.data?.isForOrdersPayment ?? false;
    this.isUserVerified =
      this.config.data?.isUserVerified ||
      this.isUserProfileVerified(this.dataUser);
  }

  limitForAddBankingCardSecond: string;
  subscriptions = new Subscription();
  isLoading: boolean = false;
  cardNumber: string;
  isFromBankAccount: boolean = true;
  isForOrdersPayment: boolean = true;
  dataUser: UserProfile = null;
  isUserVerified = false;
  private destroy$ = new Subject<void>();

  ngOnInit() {
    setAutoFocous('p-inputtext');
    this._translate
      .get('limitForAddBankingCardSecond')
      .subscribe((response) => {
        this.limitForAddBankingCardSecond = response;
      });
  }

  private isUserProfileVerified(user?: UserProfile | null): boolean {
    if (!user) return false;
    const hasName = !!user.name && user.name.trim().length > 0;
    const hasFamily = !!user.family && user.family.trim().length > 0;
    const hasNational =
      !!user.nationalCode && user.nationalCode.trim().length === 10;
    const hasGender =
      user.gender !== null && user.gender !== undefined && user.gender !== null;
    const hasBirth = !!user.birthDate;
    return hasName && hasFamily && hasNational && hasGender && hasBirth;
  }

  ngAfterViewInit() {}

  submit() {
    if (!this.isUserVerified && !this.isForOrdersPayment) {
      return;
    }
    this.isLoading = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .post(Address.BankAccountSave, { cardNumber: this.cardNumber })
        .pipe(
          finalize(() => {
            this.isLoading = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (res) => {
            if (res) {
              this.ref.close(res.data);
              this.cdr.detectChanges();
            }
          },
          (error) => {
            if (error.status === 466) {
              this.cdr.markForCheck();
            }
          }
        )
    );
  }

  onSaveBaseInfo() {
    this.isUserVerified = true;
    this.cdr.detectChanges();
  }

  close() {
    this.ref.close(null);
  }
  ngOnDestroy() {
    this.subscriptions.unsubscribe();
    this.destroy$.next();
    this.destroy$.complete();
  }
}
