import { AlertHelper } from '../../../core/helpers/alert.helper';
import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  EventEmitter,
  Input,
  OnChanges,
  OnDestroy,
  OnInit,
  Output,
  ViewChild,
} from '@angular/core';
import { stringIsEmptyOrNull } from '@helpers/validator.helper';
import {
  CountdownConfig,
  CountdownComponent,
  CountdownEvent,
} from 'ngx-countdown';
import { finalize, Subscription } from 'rxjs';
import { response } from '@interfaces/response';
import { UserStore } from '@stores/user.store';
import { CommonService } from '@services/common.service';
import { Address } from '@enums/address.enum';
import { UserProfile } from '@interfaces/user/userProfile';
import { VerifyModel } from 'src/app/core/params/auth/verifyMode';
import { fadeAnimation } from 'src/app/core/animations/animations';
import { toEnglishDigits } from '@helpers/convertToEnglishDigit.helper';
import { CountryModel } from '@models/orders/CountryModel';
import { FileUploadService } from '@services/fileUpload.service';
import { CaptchaComponent } from '@components/captcha/captcha.component';
import { AuthStore } from '@stores/AuthSetting.store';

@Component({
  selector: 'app-mobile-verification',
  templateUrl: './mobile-verification.component.html',
  styleUrls: ['./mobile-verification.component.scss'],
  changeDetection: ChangeDetectionStrategy.OnPush,
  animations: [fadeAnimation],
  host: { '[@fadeInAnimation]': '' },
})
export class MobileVerificationComponent
  implements OnInit, OnChanges, OnDestroy
{
  constructor(
    private _alert: AlertHelper,
    private _commonService: CommonService,
    private _userStore: UserStore,
    private cdr: ChangeDetectorRef,
    private _fileUploadService: FileUploadService,
    private _authSettingStore: AuthStore
  ) {
    this._userStore.user.subscribe((res) => {
      if (res != null) this.dataUserFullDetails = res;
      this.isLoadingGet = false;
    });
    this._userStore.getUserBehavior().subscribe((res) => {
      if (res != null) {
        this.isLoadingGet = false;
        this.dataUserFullDetails = res;
      }
    });

    this._authSettingStore.isActiveCaptcha.subscribe((res) => {
      if (res != null) {
        this.isActiveCaptcha = res;
      }
    });

    this._authSettingStore.isActiveCaptchaBehavior.subscribe((res) => {
      if (res != null) {
        this.isActiveCaptcha = res;
      }
    });
    this._authSettingStore.isActiveWhatsapp.subscribe((res) => {
      if (res != null) {
        this.isActiveWhatsapp = res;
      }
    });

    this._authSettingStore.isActiveWhatsappBehavior.subscribe((res) => {
      if (res != null) {
        this.isActiveWhatsapp = res;
      }
    });

    this._authSettingStore.isActiveBotTelegram.subscribe((res) => {
      if (res != null) {
        this.isActiveBotTelegram = res;
      }
    });

    this._authSettingStore.isActiveBotTelegramBehavior.subscribe((res) => {
      if (res != null) {
        this.isActiveBotTelegram = res;
      }
    });

    this._authSettingStore.parallelSendingWhatsappLogin.subscribe((res) => {
      if (res != null) {
        this.parallelSendingWhatsappLogin = res;
      }
    });

    this._authSettingStore.parallelSendingWhatsappLoginBehavior.subscribe(
      (res) => {
        if (res != null) {
          this.parallelSendingWhatsappLogin = res;
        }
      }
    );

    this._authSettingStore.parallelSendingTelegramLogin.subscribe((res) => {
      if (res != null) {
        this.parallelSendingTelegramLogin = res;
        Promise.resolve().then(() => this.cdr.detectChanges());
      }
    });

    this._authSettingStore.parallelSendingTelegramLoginBehavior.subscribe(
      (res) => {
        if (res != null) {
          this.parallelSendingTelegramLogin = res;
          Promise.resolve().then(() => this.cdr.detectChanges());
        }
      }
    );
  }
  isShow = true;
  dataUserFullDetails: UserProfile;
  isActiveCaptcha: boolean;
  isActiveWhatsapp: boolean = true;
  isActiveBotTelegram: boolean = true;
  parallelSendingWhatsappLogin: boolean;
  parallelSendingTelegramLogin: boolean;
  model: VerifyModel = new VerifyModel();
  mobile: string;
  isEmptyMobile: boolean;
  isLoading: boolean = false;
  isLoadingResendCode: boolean = false;
  canResend: boolean = false;
  isShowOtp: boolean = false;
  setInterval = setInterval;
  isLoadingGet: boolean = false;
  isLoadingVerify: boolean = false;
  otp: string;
  isLoadingGetCountry: boolean = false;
  selectedCountry!: CountryModel;
  isLodingGetUrl: boolean = false;
  isLoadingConnect: boolean = false;
  isLoadingDisconnect: boolean = false;
  dataCountry: CountryModel[] = [];
  @Output() onSave = new EventEmitter();
  @Input() data: UserProfile = null;
  @Input() isCompleteMobile: boolean = null;
  @Input() isForAlarmOrders: boolean = false;
  @Output() reloadUserInfo = new EventEmitter<void>();
  subscriptions = new Subscription();
  typeSendCodeToMobile: 'sms' | 'whatsapp' | 'telegram' = 'sms';
  sendCodeToWhatsApp: boolean = false;
  sendCodeToTelegram: boolean = false;
  isEditting: boolean = false;
  mobileOld: string;
  @ViewChild('captchaMobile') captcha: CaptchaComponent;
  @ViewChild('cd', { static: false }) private countdown: CountdownComponent;
  config: CountdownConfig = {
    leftTime: 0,
    format: 'm:s',
  };

  setNewTimer() {
    this.config = {
      leftTime: 120,
      format: 'm:s',
    };
    this.canResend = false;
    this.cdr.detectChanges();
  }

  finishTimer(e: CountdownEvent) {
    if (e.action === 'done') {
      this.canResend = true;
      this.isShowOtp = false;
      if (this.isActiveCaptcha && this.captcha) {
        this.captcha.handleRefreshCaptcha();
      }
    }
    this.cdr.detectChanges();
  }

  cancel() {
    this.isEditting = false;
    this.isShowOtp = false;
    if (this.mobileOld) {
      this.mobile = this.mobileOld;
      this.data.mobile = this.mobileOld;
    }
    this.canResend = !this.data?.mobile;
    this.cdr.detectChanges();
  }

  ngOnInit(): void {
    this.getCountry();
  }

  ngOnDestroy() {
    this.subscriptions.unsubscribe();
  }
  ngAfterViewInit() {}

  ngOnChanges() {
    if (this.data != undefined && this.data != null) {
      this.mobile = this.data.mobile;
      this.isEditting = false;
      this.canResend = !this.data.mobile;
      Promise.resolve().then(() => this.cdr.detectChanges());
    }
  }

  conneectToTelegram() {
    this.isLoadingConnect = true;
    this.subscriptions.add(
      this._commonService
        .post(Address.ConnectToTelegram, {}, false)
        .pipe(
          finalize(() => {
            this.isLoadingConnect = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            window.open(response.data, '_blank');
          },
          (error) => {}
        )
    );
  }

  disconneectToTelegram() {
    this.isLoadingDisconnect = true;
    this.subscriptions.add(
      this._commonService
        .delete(Address.DiscconnectToTelegram, {})
        .pipe(
          finalize(() => {
            this.isLoadingDisconnect = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this.reloadUserInfo.emit();
            this._alert.success(
              'حذف اتصال حساب شما از ربات تلگرام با موفقیت انجام گردید'
            );
          },
          (error) => {}
        )
    );
  }

  editInfo() {
    this.mobileOld = this.mobile;
    this.isEditting = true;
    this.canResend = true;
    this.isShowOtp = false;
    this.cdr.detectChanges();
  }

  userGetInfo() {
    this.isLoadingGet = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(Address.UserGet, {}, false)
        .pipe(
          finalize(() => {
            this.isLoadingGet = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this.data = response.data;
            this.mobile = this.data.mobile;
            this._userStore.user.next(response.data);
            this._userStore.userBehavior.next(response.data);
          },
          (error) => {}
        )
    );
  }

  verify() {
    if (stringIsEmptyOrNull(this.otp)) {
      this._alert.warning('لطفا کد یکبار مصرف را وارد کنید');
      return;
    }
    this.isLoadingVerify = true;
    this.cdr.detectChanges();
    const payload = {
      code: this.otp,
      typeVerify: 101,
      tokenVerify: this.model.token,
      address: this.model.address,
      prefixPhone: this.selectedCountry.prefixPhone,
      countryId: this.selectedCountry.countryId,
    };
    const obs$ = this._commonService
      .post(Address.VerifyInPanel, payload, false)
      .pipe(
        finalize(() => {
          this.isLoadingVerify = false;
          this.cdr.detectChanges();
        })
      );
    const sub = obs$.subscribe({
      next: (response: any) => {
        this.isShowOtp = false;
        this.canResend = true;
        this.data.mobile = this.mobile;
        this.isEditting = false;
        this.otp = null;
        this.onSave.emit(true);
        this._alert.success('شماره همراه شما با موفقیت تایید شد');
        this.cdr.detectChanges();
        if (!this.isForAlarmOrders) {
          this.userGetInfo();
        }
      },
      error: (err: any) => {},
    });

    this.subscriptions.add(sub);
  }

  submit() {
    if (stringIsEmptyOrNull(this.mobile)) {
      this._alert.warning('لطفا شماره همراه خود را وارد کنید');
      return;
    }

    if (
      this.isActiveCaptcha &&
      (!this.captcha ||
        !this.captcha.CaptchaCode ||
        stringIsEmptyOrNull(this.captcha.CaptchaCode.value))
    ) {
      this._alert.warning('لطفا کد امنیتی را وارد کنید');
      return;
    }

    this.sendCodeToWhatsApp = this.typeSendCodeToMobile === 'whatsapp';
    this.sendCodeToTelegram = this.typeSendCodeToMobile === 'telegram';
    this.mobile = toEnglishDigits(this.mobile);
    this.isLoading = true;
    this.cdr.detectChanges();

    const payload = {
      address: String(this.mobile),
      typeVerify: 101,
      sendCodeToWhatsApp: this.sendCodeToWhatsApp,
      sendCodeToTelegram: this.sendCodeToTelegram,
      prefixPhone: this.selectedCountry?.prefixPhone,
      countryId: this.selectedCountry?.countryId,
      sendCodeLandlineNumber: false,
      captchaCode: this.captcha?.captchaInfo?.captchaCode,
      captchaId: this.captcha?.captchaInfo?.captchaId,
    };

    const obs$ = this._commonService
      .post(Address.SendInPanel, payload, false)
      .pipe(
        finalize(() => {
          this.isLoading = false;
          this.cdr.detectChanges();
        })
      );

    const sub = obs$.subscribe({
      next: (response: any) => {
        setTimeout(() => this.setNewTimer(), 1000);
        this.isShowOtp = true;
        this.model.token = response.data?.token;
        this.model.typeVerify = response.data?.typeVerify;
        this.model.address = response.data?.address;

        if (this.sendCodeToTelegram || this.sendCodeToWhatsApp) {
          this._alert.success('کد تایید برای شما ارسال شد');
        } else {
          this._alert.success('بزودی پیامک تایید برای شما ارسال خواهد شد');
        }
        this.cdr.detectChanges();
      },
      error: (err: any) => {
        if (this.isActiveCaptcha && this.captcha) {
          try {
            this.captcha.handleRefreshCaptcha();
          } catch (e) {}
        }
      },
    });

    this.subscriptions.add(sub);
  }

  onCountrySelected(country: any) {
    this.selectedCountry = country;
    this.model.prefixPhone = country?.prefixPhone ?? null;
  }

  getCountry() {
    this.isLoadingGetCountry = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(
          Address.CountryGetAll,
          { useInOrder: null, useInRegister: true },
          false
        )
        .pipe(
          finalize(() => {
            this.isLoadingGetCountry = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response) => {
            this.dataCountry = response.data.filter(
              (item) => item.isActive && !stringIsEmptyOrNull(item.prefixPhone)
            );
            for (let country of this.dataCountry) {
              if (this.dataUserFullDetails.prefixPhone) {
                if (country.countryId == this.dataUserFullDetails.countryId) {
                  this.selectedCountry = country;
                }
              } else if (country.isoCode == 'IRN') {
                this.selectedCountry = country;
              }
            }
            this._fileUploadService
              .fileUrlGeneratorPutCountryDetection(this.dataCountry, 'avatarId')
              .subscribe({
                next: (result) => {
                  this.dataCountry = [...this.dataCountry];
                  this.cdr.detectChanges();
                },
                error: (error) => {
                  this.cdr.detectChanges();
                },
              });
            this.cdr.detectChanges();
          },
          (error) => {}
        )
    );
  }

  trackByCountry(index: number, item: any) {
    return item.countryId;
  }
}
