import { stringIsEmptyOrNull } from './../../../core/helpers/validator.helper';
import { AlertHelper } from '@helpers/alert.helper';
import {
  AfterViewInit,
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  EventEmitter,
  Input,
  OnChanges,
  OnDestroy,
  OnInit,
  Output,
} from '@angular/core';
import { Router } from '@angular/router';
import { Address } from '@enums/address.enum';
import { UpdateProfileModel } from '@models/user/UpdateProfileModel';
import { CommonService } from '@services/common.service';
import { ConfirmationService } from 'primeng/api';
import { Subscription } from 'rxjs';
import { finalize } from 'rxjs/operators';
import { response } from '@interfaces/response';
import { genders } from '@helpers/values.helper';
import { UserStore } from '@stores/user.store';
import { cloneDeep } from 'lodash-es';
import { fadeAnimation } from 'src/app/core/animations/animations';
import { isAlphabetic } from '@helpers/isAlphabetic.helper';
import { UserProfile } from '@interfaces/user/userProfile';

@Component({
  selector: 'app-update-profile',
  templateUrl: './update-profile.component.html',
  styleUrls: ['./update-profile.component.scss'],
  changeDetection: ChangeDetectionStrategy.OnPush,
  animations: [fadeAnimation],
  host: { '[@fadeInAnimation]': '' },
})
export class UpdateProfileComponent
  implements OnDestroy, OnChanges, OnInit, AfterViewInit
{
  isLoading: boolean = false;
  isLoadingGet: boolean = false;
  subscriptions = new Subscription();
  isLockUpdate: boolean = false;
  profileData: UserProfile = new UserProfile();
  @Output() onSave = new EventEmitter();
  @Input() data: UserProfile = new UserProfile();
  @Input() isFromBankAccount: boolean = false;
  @Input() isForAlarmOrders: boolean = false;
  gender = genders();

  constructor(
    private _commonService: CommonService,
    private _router: Router,
    private confirmationService: ConfirmationService,
    private _alert: AlertHelper,
    private _userStore: UserStore,
    private cdr: ChangeDetectorRef
  ) {}

  ngOnInit(): void {
    this._userStore.getUserBehavior().subscribe((res) => {
      this.data = res;
      this.isLockUpdate = res.isLockUpdate;
    });
  }

  ngAfterViewInit(): void {
    this.cdr.detectChanges();
  }

  ngOnDestroy() {
    this.subscriptions.unsubscribe();
  }

  ngOnChanges() {
    if (
      this.data != undefined &&
      this.data != null &&
      !stringIsEmptyOrNull(this.data.nationalCode)
    )
      this.profileData = cloneDeep(this.data);
    this.data = cloneDeep(this.profileData);
  }

  confirm(alarm: string) {
    this.confirmationService.confirm({
      message: alarm,
      key: 'alarm',
      accept: () => {
        this._router.navigateByUrl('/');
      },
    });
  }

  userGetInfo() {
    this.isLoadingGet = true;
    this.subscriptions.add(
      this._commonService
        .put(Address.UserGet, {}, false)
        .pipe(
          finalize(() => {
            this.isLoadingGet = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this.profileData = cloneDeep(response.data);

            // this._userStore.user.next({
            //   ...this.data,
            //   name: this.profileData.name,
            //   family: this.profileData.family,
            //   isLockUpdate: this.profileData.isLockUpdate,
            //   birthDate: this.profileData.birthDate,
            //   nationalCode: this.profileData.nationalCode,
            // });
            // this._userStore.userBehavior.next({
            //   ...this.data,
            //   name: this.profileData.name,
            //   family: this.profileData.family,
            //   isLockUpdate: this.profileData.isLockUpdate,
            //   birthDate: this.profileData.birthDate,
            //   nationalCode: this.profileData.nationalCode,
            // });
          },
          (error) => {}
        )
    );
  }

  submit() {
    if (stringIsEmptyOrNull(this.data.birthDate)) {
      this._alert.warning('لطفا تاریخ تولد خود را وارد کنید');
      return;
    }
    let isAlphabeticName = isAlphabetic(this.data.name.replace(/\s/g, ''));
    let isAlphabeticFamily = isAlphabetic(this.data.family.replace(/\s/g, ''));
    if (isAlphabeticName && isAlphabeticFamily) {
      this.isLoading = true;
      if (this.data.gender != null) {
        this.profileData.gender = Number(this.data.gender);
      }
      this.profileData.nationalCode = this.data.nationalCode;
      this.profileData.birthDate = this.data.birthDate;
      this.profileData.name = this.data.name;
      this.profileData.family = this.data.family;
      this.profileData.isLockUpdate = true;
      this.subscriptions.add(
        this._commonService
          .put(Address.UpdateProfile, this.profileData, false)
          .pipe(
            finalize(() => {
              this.isLoading = false;
              this.cdr.detectChanges();
            })
          )
          .subscribe(
            (response: response) => {
              if (!this.isForAlarmOrders) {
                this.userGetInfo();
              }
              this.onSave.emit(true);
              this._alert.success('ثبت اطلاعات با موفقیت انجام شد');
              this.isLockUpdate = true;
            },
            (error) => {
              this.profileData = new UpdateProfileModel();
            }
          )
      );
    } else {
      this._alert.warning('مقدار عددی در نام و نام خانوادگی پذیرفته نیست');
    }
  }
}
