import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  OnInit,
} from '@angular/core';
import { Address } from '@enums/address.enum';
import { AlertHelper } from '@helpers/alert.helper';
import { isArabic } from '@helpers/validator.helper';
import { response } from '@interfaces/response';
import { CommonService } from '@services/common.service';
import { UserStore } from '@stores/user.store';
import { Subscription, finalize } from 'rxjs';

@Component({
  selector: 'cash-exchange-wallet',
  templateUrl: 'cashExchange.component.html',
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class CashExchangeWalletComponent implements OnInit {
  isLoading: boolean;
  isValue: boolean;
  exchangePrice: number;
  formattedValue: string = '';
  walletAmount: number;
  subscriptions = new Subscription();
  constructor(
    private _commonService: CommonService,
    private cdr: ChangeDetectorRef,
    private _userStore: UserStore,
    private _alert: AlertHelper
  ) {}

  ngOnInit() {
    this._userStore.getUser().subscribe((res) => {
      if (res.walletId) {
        this.walletAmount = Number(res.amountWallet);
      }
      this.cdr.detectChanges();
    });
    this._userStore.getUserBehavior().subscribe((res) => {
      if (res.walletId) {
        this.walletAmount = Number(res.amountWallet);
      }
      this.cdr.detectChanges();
    });
  }

  ngOnDestroy() {
    this.subscriptions.unsubscribe();
  }

  getValue(inputElement: HTMLInputElement): void {
    this.exchangePrice = parseFloat(
      inputElement.dataset['numericValue'] || '0'
    );
  }

  checkIsArabicNumber(event: any): void {
    if (isArabic(event.value)) {
      this._alert.warning('لطفا کیبورد خود را انگلیسی کنید');
    }
  }

  valueController(event: any) {
    if (
      !this.exchangePrice ||
      this.exchangePrice == null ||
      this.exchangePrice == 0 ||
      this.exchangePrice < 1000
    ) {
      this.isValue = false;
    } else {
      this.isValue = true;
    }
  }

  navigateAcceptPage() {
    if (!this.exchangePrice) {
      this._alert.warning(
        'برای نقد موجودی کیف پول، باید عددی بالای 1000 تومان وارد نمایید.'
      );
      return;
    }
    if (this.walletAmount < this.exchangePrice) {
      this._alert.warning('مقدار درخواستی از موجودی کیف پول بیشتر میباشد.');
    } else {
      this.isLoading = true;

      this.cdr.detectChanges();
      this.subscriptions.add(
        this._commonService
          .post(
            Address.cashApplicationSave,
            {
              price: this.exchangePrice,
            },
            false
          )
          .pipe(
            finalize(() => {
              this.isLoading = false;
              this.cdr.detectChanges();
            })
          )
          .subscribe(
            (response: response) => {
              if (response) {
                window.location.href =
                  '/dashboard/user/accept-cash-exchange-wallet';
              }
            },
            (error) => {
              this.isLoading = false;
              this.cdr.detectChanges();
            }
          )
      );
    }
  }
}