import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  OnDestroy,
  OnInit,
  ViewChild,
} from '@angular/core';
import { Address } from '@enums/address.enum';
import { ColumnKey } from '@enums/columnKey.enum';
import {
  setStorageItem,
  setStorageItemCallback,
} from '@helpers/localstorage.helper';
import { response } from '@interfaces/response';
import { MethodGatewayModel } from '@models/orders/MethodGatewayModel';
import { TranslateService } from '@ngx-translate/core';
import { CommonService } from '@services/common.service';
import { ConfirmationService } from 'primeng/api';
import { DynamicDialogConfig, DynamicDialogRef } from 'primeng/dynamicdialog';
import { Subscription, finalize } from 'rxjs';
import { PurchaseOrderModel } from 'src/app/core/params/order/PurchaseOrderModel';

@Component({
  selector: 'wallet-recharge',
  templateUrl: 'wallet-recharge.component.html',
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class WalletRechargeComponent implements OnInit, OnDestroy {
  bankPorts: MethodGatewayModel;
  paymentGatewayListFinal: any[] = [];
  checkoutActivations: any;
  isLoading: boolean;
  isLoadingRedirect: boolean = false;
  rechargeValue: any;
  model: PurchaseOrderModel = new PurchaseOrderModel();
  maximumPrice: number;
  minimumPrice: number;
  suggestedValues = ['100000', '500000', '1000000', '10000000'];
  price: string = '20000';
  url: string;
  isAcceptedMobile: boolean = false;
  isAcceptedEmail: boolean = false;
  isAcceptedTel: boolean = false;
  isAcceptedNationalCard: boolean = false;
  hasBankAccount: boolean = false;
  isAcceptedSelfie: boolean = false;
  isAcceptedBasicInformation: boolean = false;
  @ViewChild('account') input: ElementRef;

  constructor(
    private _translateService: TranslateService,
    private _commonService: CommonService,
    private cdr: ChangeDetectorRef,
    public config: DynamicDialogConfig,
    private confirmationService: ConfirmationService,
    public ref: DynamicDialogRef
  ) {}

  chooseFromSuggestions(n: string) {
    this.price = n;
  }

  subscriptions = new Subscription();

  settingCheckoutGet() {
    this.isLoading = true;
    this.cdr.detectChanges();

    this.subscriptions.add(
      this._commonService
        .put(Address.SettingCheckoutGet, {})
        .pipe(
          finalize(() => {
            if (this.config.data && this.config.data.value) {
              this.url = this.config.data.url;
              if (this.config.data.value > this.maximumPrice) {
                this.price = this.maximumPrice.toString();
              } else {
                this.price = this.config.data.value;
              }
            }
          })
        )
        .subscribe((response) => {
          this.checkoutActivations = response.data;

          this._translateService
            .get('paymentGateway.data')
            .subscribe((response: any[]) => {
              let paymentGatewayList = response;

              for (let item of paymentGatewayList) {
                if (
                  Number(item.value) == 452 &&
                  this.checkoutActivations.isActiveJibitOnlinePay
                ) {
                  this.paymentGatewayListFinal.push(item);
                }
                if (
                  Number(item.value) == 453 &&
                  this.checkoutActivations.isActiveVandar
                ) {
                  this.paymentGatewayListFinal.push(item);
                }
                if (
                  Number(item.value) == 454 &&
                  this.checkoutActivations.isActiveShepaPPG
                ) {
                  this.paymentGatewayListFinal.push(item);
                }
                if (
                  Number(item.value) == 455 &&
                  this.checkoutActivations.isActiveZarinpal
                ) {
                  this.paymentGatewayListFinal.push(item);
                }
                if (
                  Number(item.value) == 457 &&
                  this.checkoutActivations.isActiveZarinLink
                ) {
                  this.paymentGatewayListFinal.push(item);
                }
                if (
                  Number(item.value) == 458 &&
                  this.checkoutActivations.isActiveSepal
                ) {
                  this.paymentGatewayListFinal.push(item);
                }
              }

              for (let item of this.paymentGatewayListFinal) {
                if (
                  this.checkoutActivations.isDefaultJibit &&
                  item.value === '452'
                ) {
                  this.bankPorts = item;
                }
                if (
                  this.checkoutActivations.isDefaultVandar &&
                  item.value === '453'
                ) {
                  this.bankPorts = item;
                }
                if (
                  this.checkoutActivations.isDefaultShepa &&
                  item.value === '454'
                ) {
                  this.bankPorts = item;
                }
                if (
                  this.checkoutActivations.isDefaultZarinpal &&
                  item.value === '455'
                ) {
                  this.bankPorts = item;
                }
                if (
                  this.checkoutActivations.isDefaultZarinLink &&
                  item.value === '457'
                ) {
                  this.bankPorts = item;
                }
                if (
                  this.checkoutActivations.isDefaultSepal &&
                  item.value === '458'
                ) {
                  this.bankPorts = item;
                }
              }

              this.isLoading = false;
              this.cdr.detectChanges();
            });
        })
    );
  }

  ngOnInit() {
    this.settingCheckoutGet();
  }


  ngOnDestroy() {
    this.subscriptions.unsubscribe();
  }

  payByPortalType() {
    switch (this.bankPorts.value) {
      case '452':
        this.chargeWalletByJibit();
        break;
      case '453':
        this.chargeWalletByVandar();
        break;
      case '454':
        this.chargeWalletByShepa();
        break;
      case '455':
        this.chargeWalletByZarin();
        break;
      case '457':
        this.chargeWalletByZarinLink();
        break;
      case '458':
        this.chargeWalletBySepal();
        break;
      default:
        break;
    }
  }

  chargeWalletByJibit() {
    this.isLoadingRedirect = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .post(
          Address.ChargeWalletJBByUserSave,
          { paidAmount: this.price, callbackUrl: this.url },
          false
        )
        .pipe(finalize(() => {}))
        .subscribe(
          (res: response) => {
            setStorageItem(ColumnKey.CallbackUrl, this.url);

            window.location.href = res.data.redirectUrl;
          },

          (error) => {
            this.isLoadingRedirect = false;
            this.cdr.detectChanges();
          }
        )
    );
  }

  chargeWalletByZarin() {
    this.isLoadingRedirect = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .post(
          Address.ChargeWalletZarinByUserSave,
          { paidAmount: this.price, callbackUrl: this.url },
          false
        )
        .pipe(finalize(() => {}))
        .subscribe(
          (res: response) => {
            setStorageItem(ColumnKey.CallbackUrl, this.url);
            window.location.href = res.data.redirectUrl;
          },

          (error) => {
            this.isLoadingRedirect = false;
            this.cdr.detectChanges();
          }
        )
    );
  }

  chargeWalletByZarinLink() {
    this.isLoadingRedirect = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .post(
          Address.ChargeWalletZarinLinkByUserSave,
          { paidAmount: this.price },
          false
        )
        .pipe(finalize(() => {}))
        .subscribe(
          (res: response) => {
            setStorageItemCallback(
              ColumnKey.PaidCode,
              res.data.paidCode,
              () => {
                window.location.href = res.data.redirectUrl;
              }
            );
          },

          (error) => {
            this.isLoadingRedirect = false;
            this.cdr.detectChanges();
          }
        )
    );
  }

  chargeWalletBySepal() {
    this.isLoadingRedirect = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .post(
          Address.ChargeWalletSepalByUserSave,
          { paidAmount: this.price },
          false
        )
        .pipe(finalize(() => {}))
        .subscribe(
          (res: response) => {
            setStorageItemCallback(
              ColumnKey.PaidCode,
              res.data.paidCode,
              () => {
                window.location.href = res.data.redirectUrl;
              }
            );
          },

          (error) => {
            this.isLoadingRedirect = false;
            setTimeout(() => {
              if (error.error.message) {
                const userLocationStr = localStorage.getItem('user-location');
                if (userLocationStr) {
                  try {
                    const userLocation = JSON.parse(userLocationStr);
                    if (
                      userLocation.countryCode &&
                      userLocation.countryCode !== 'IR'
                    ) {
                      localStorage.removeItem('user-location');
                      this.refreshConfirm();
                    }
                  } catch (e) {
                    localStorage.removeItem('user-location');
                    this.refreshConfirm();
                  }
                }
              }
            }, 100);
            this.cdr.detectChanges();
          }
        )
    );
  }

  refreshConfirm() {
    this.confirmationService.confirm({
      message:
        this._translateService.translations[this._translateService.currentLang][
          'refreshPageMessage'
        ],
      header:
        this._translateService.translations[this._translateService.currentLang][
          'checkVpnConnection'
        ],
      key: 'refreshPage',
      acceptLabel:
        this._translateService.translations[this._translateService.currentLang][
          'refreshPage'
        ],
      rejectVisible: false,
      acceptIcon: 'none',
      accept: () => {
        location.reload();
      },
    });
  }

  chargeWalletByShepa() {
    this.isLoadingRedirect = true;
    this.cdr.detectChanges();

    this.subscriptions.add(
      this._commonService
        .post(
          Address.ChargeWalletShepaByUserSave,
          { paidAmount: this.price, callbackUrl: this.url },
          false
        )
        .pipe(finalize(() => {}))
        .subscribe(
          (res: response) => {
            this.isLoadingRedirect = false;
            setStorageItem(ColumnKey.CallbackUrl, this.url);
            window.location.href = res.data.redirectUrl;
          },

          (error) => {
            this.isLoadingRedirect = false;
            this.cdr.detectChanges();
          }
        )
    );
  }

  chargeWalletByVandar() {
    this.isLoadingRedirect = true;
    this.cdr.detectChanges();

    this.subscriptions.add(
      this._commonService
        .post(
          Address.ChargeWalletVNByUserSave,
          { paidAmount: this.price, callbackUrl: this.url },
          false
        )
        .pipe(finalize(() => {}))
        .subscribe(
          (res: response) => {
            this.isLoadingRedirect = false;
            setStorageItem(ColumnKey.CallbackUrl, this.url);
            window.location.href = res.data.redirectUrl;
          },

          (error) => {
            this.isLoadingRedirect = false;
            this.cdr.detectChanges();
          }
        )
    );
  }
}
