import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  EventEmitter,
  Input,
  OnDestroy,
  OnInit,
  Output,
} from '@angular/core';
import { ActivatedRoute, Router } from '@angular/router';
import { Address } from '@enums/address.enum';
import { MessageStatic } from '@enums/messageStatic.enum';
import { CommonService } from '@services/common.service';
import { Location } from '@angular/common';
import { cloneDeep } from 'lodash-es';
import { Message } from 'primeng/api';
import { DialogService } from 'primeng/dynamicdialog';
import { Subject, Subscription, finalize, takeUntil } from 'rxjs';
import { OrderSearchModel } from 'src/app/core/params/order/OrderSearchModel';
import { BaseParams } from 'src/app/core/params/user/baseParams';
import { fadeAnimation } from 'src/app/core/animations/animations';
import { WalletRechargeComponent } from './wallet-rechage/wallet-recharge.component';
import { WalletModelGet } from '@models/user/WalletModelGet';
import { TableLazyLoadEvent } from 'primeng/table';
import { PagingHelper } from '@helpers/paging.helper';
import { CashExchangeWalletComponent } from './cash-exchange/cashExchange.component';
import { UserProfile } from '@interfaces/user/userProfile';
import { UserStore } from '@stores/user.store';
import { response } from '@interfaces/response';
import { AlertHelper } from '@helpers/alert.helper';
import { TranslateService } from '@ngx-translate/core';
import { VerifyingModel } from '@models/user/VerifyingModel';
import { WalletSettingModel } from '@models/user/walletSettingModel';
import { ModalContentMobileComponent } from '../../components/modalContentMobile/modalContentMobile.component';

@Component({
  selector: 'wallet',
  templateUrl: './wallet.component.html',
  styleUrls: ['./wallet.component.scss'],
  changeDetection: ChangeDetectionStrategy.OnPush,
  animations: [fadeAnimation],
  host: { '[@fadeInAnimation]': '' },
})
export class WalletComponent implements OnInit, OnDestroy {
  constructor(
    private _commonService: CommonService,
    private cdr: ChangeDetectorRef,
    public dialogService: DialogService,
    private _router: Router,
    public _activatedroute: ActivatedRoute,
    public _pagingHelper: PagingHelper,
    public location: Location,
    private _userStore: UserStore,
    private _messageService: AlertHelper,
    private _translateService: TranslateService
  ) {}

  subscriptions = new Subscription();
  isloadingGetWalletSetting: boolean;
  messages: Message[] = [];
  filter: any = {};
  @Input() walletTransactions: WalletModelGet[] = [];
  @Input() totalRecords: number = 0;
  @Output() isVerifiedInfo: EventEmitter<boolean> = new EventEmitter<boolean>();
  isLoading: boolean = false;
  query: BaseParams;
  overlayPanel: boolean;
  search: OrderSearchModel = new OrderSearchModel();
  dataUser: UserProfile = new UserProfile();
  walletId: string;
  walletAmount: number;
  message: string = MessageStatic.NotFound;
  rowsPerPage = this._pagingHelper.rowsPersPage();
  isLoadingActivate: boolean;
  loadingGetInfo: boolean;
  dataVerifying: VerifyingModel[] = [];
  private destroy$ = new Subject<void>();
  isAcceptedMobile: boolean = false;
  hasBankAccount: boolean = false;
  isAcceptedEmail: boolean = false;
  isAcceptedBasicInformation: boolean = false;
  isAcceptedTel: boolean = false;
  isAcceptedSelfie: boolean = false;
  isAcceptedNationalCard: boolean = false;
  dataSettingWallet: WalletSettingModel = new WalletSettingModel();
  styleClass: string = 'verifyingModal';
  width: string = '50%';
  height: string = '95%';
  header: string = 'احراز اطلاعات';

  ngOnDestroy(): void {
    this.subscriptions.unsubscribe();
    this.destroy$.next();
    this.destroy$.complete();
  }

  ngOnInit(): void {
    this.getWalletSetting();
    this.query = new BaseParams();
    this._userStore
      .getUserBehavior()
      .pipe(takeUntil(this.destroy$))
      .subscribe((res) => {
        if (res) {
          this.dataUser = res;
          this.walletId = res.walletId;
          this.walletAmount = Number(res.amountWallet);
        }
        this._userStore
          .getUserVerifyingBehavior()
          .pipe(takeUntil(this.destroy$))
          .subscribe((res) => {
            if (res) {
              this.dataVerifying = res;
              this.checkUserIsVerified();
            }
          });
      });

    this.cdr.detectChanges();
  }

  userGetInfo() {
    this.loadingGetInfo = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .put(Address.UserGet, {}, false)
        .pipe(
          finalize(() => {
            this.loadingGetInfo = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response: response) => {
            this.dataUser = response.data;
            this._userStore.user.next(this.dataUser);
            this._userStore.userBehavior.next(this.dataUser);
            this.walletId = this.dataUser.walletId;
            this.cdr.detectChanges();
          },
          (error) => {}
        )
    );
  }

  goBackToPrevPage() {
    this.location.back();
  }

  activateWallet() {
    this.isLoadingActivate = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .post(Address.WalletCreateByCustomer, {})
        .pipe(
          finalize(() => {
            this.isLoadingActivate = false;
          })
        )
        .subscribe((res) => {
          this.userGetInfo();
          this.cdr.detectChanges();
          this._messageService.success(
            this._translateService.translations[
              this._translateService.currentLang
            ]['yourWalletActivatedSuccessfully']
          );
        })
    );
  }

  getWalletSetting() {
    this.isloadingGetWalletSetting = true;
    this.cdr.detectChanges();
    this.subscriptions.add(
      this._commonService
        .get(Address.GetWalletSetting, {})
        .pipe(
          finalize(() => {
            this.isloadingGetWalletSetting = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe(
          (response) => {
            this.dataSettingWallet = response.data;
            this.cdr.detectChanges();
          },
          (err: any) => {},
          () => {}
        )
    );
  }

  checkUserIsVerified() {
    if (this.dataUser && this.dataVerifying) {
      for (let item of this.dataVerifying) {
        if (
          item.typeVerifying == 111 &&
          item.isAccept &&
          this.dataUser.mobile
        ) {
          this.isAcceptedMobile = true;
        }
        if (item.typeVerifying == 112 && item.isAccept && this.dataUser.email) {
          this.isAcceptedEmail = true;
        }
        if (item.typeVerifying == 113 && item.isAccept && this.dataUser.tel) {
          this.isAcceptedTel = true;
        }
        if (item.typeVerifying == 114 && item.isAccept) {
          this.isAcceptedNationalCard = true;
        }
        if (item.typeVerifying == 115 && item.isAccept) {
          this.hasBankAccount = true;
        }
        if (item.typeVerifying == 116 && item.isAccept) {
          this.isAcceptedSelfie = true;
        }
        if (
          item.typeVerifying == 118 &&
          item.isAccept &&
          this.dataUser.isLockUpdate
        ) {
          this.isAcceptedBasicInformation = true;
        }
      }
    }
  }

  async checkWalletRequirementsBeforeAction(actionCallback: Function) {
    if (!this.dataSettingWallet) {
      await this.getWalletSetting();
    }

    const missingVerifications = [];

    if (
      this.dataSettingWallet.isActiveAuthenticationWalletMobile &&
      !this.isAcceptedMobile
    ) {
      missingVerifications.push('mobile');
    }
    if (
      this.dataSettingWallet.isActiveAuthenticationWalletEmail &&
      !this.isAcceptedEmail
    ) {
      missingVerifications.push('email');
    }
    if (
      this.dataSettingWallet.isActiveAuthenticationWalletTel &&
      !this.isAcceptedTel
    ) {
      missingVerifications.push('tel');
    }
    if (
      this.dataSettingWallet.isActiveAuthenticationWalletNationalCard &&
      !this.isAcceptedNationalCard
    ) {
      missingVerifications.push('nationalCard');
    }
    if (
      this.dataSettingWallet.isActiveAuthenticationWalletSelfiImage &&
      !this.isAcceptedSelfie
    ) {
      missingVerifications.push('selfie');
    }
    if (
      this.dataSettingWallet.isActiveAuthenticationWalletBasicInformation &&
      !this.isAcceptedBasicInformation
    ) {
      missingVerifications.push('basicInfo');
    }
    if (
      this.dataSettingWallet.isActiveAuthenticationWalletBankCard &&
      !this.hasBankAccount
    ) {
      missingVerifications.push('bankCard');
    }

    if (missingVerifications.length === 0) {
      actionCallback();
      return;
    }

    const orderSetting = {
      checkMobile: missingVerifications.includes('mobile'),
      checkEmail: missingVerifications.includes('email'),
      checkPhone: missingVerifications.includes('tel'),
      checkNationalCard: missingVerifications.includes('nationalCard'),
      checkSelfiImage: missingVerifications.includes('selfie'),
      checkBasicInformation: missingVerifications.includes('basicInfo'),
      checkBankAccount: missingVerifications.includes('bankCard'),
    };

    const ref = this.modalGetMobile(orderSetting, missingVerifications);
    ref.onClose.subscribe((verified) => {
      if (verified) {
        this.userGetInfo();
        actionCallback();
        this.isVerifiedInfo.emit(true);
      }
    });
  }

  modalGetMobile(orderSetting, missingFields) {
    const ref = this.dialogService.open(ModalContentMobileComponent, {
      header: this.header,
      height: this.height,
      width: this.width,
      styleClass: this.styleClass,
      data: {
        orderSetting: orderSetting,
        missingFields: missingFields,
      },
      dismissableMask: true,
    });

    return ref;
  }

  getData(event: TableLazyLoadEvent = null): void {
    this.isLoading = true;
    this.cdr.detectChanges();
    this.query = new BaseParams();
    Object.assign(this.query, this.search);
    this.query = this._pagingHelper.requestParamsSet(event, this.query);
    this.subscriptions.add(
      this._commonService
        .put(Address.TransActionGetAll, { transactionTypeList: '683,684' })
        .pipe(
          finalize(() => {
            this.isLoading = false;
            this.cdr.detectChanges();
          })
        )
        .subscribe((res) => {
          if (res && res.data) {
            this.walletTransactions = cloneDeep(res.data);
            this.totalRecords = res.actualSize;
            this.cdr.detectChanges();
          }
        })
    );
  }

  toggle() {
    this.overlayPanel = !this.overlayPanel;
  }

  navigateToCashExchange() {
    this._router.navigateByUrl('/dashboard/user/user-requests');
  }

  walletRecharge() {
    this.checkWalletRequirementsBeforeAction(() => {
      const ref = this.dialogService.open(WalletRechargeComponent, {
        header:
          this._translateService.translations[
            this._translateService.currentLang
          ]['walletRecharge'],
        width: '23%',
        styleClass: 'wallet-modal',
        dismissableMask: true,
      });
    });
  }

  cashExchangeWallet() {
    this.checkWalletRequirementsBeforeAction(() => {
      const ref = this.dialogService.open(CashExchangeWalletComponent, {
        header:
          this._translateService.translations[
            this._translateService.currentLang
          ]['walletCash'],
        height: '70%',
        width: '40%',
        styleClass: 'wallet-modal',
        dismissableMask: true,
      });
    });
  }
}
