﻿using Microsoft.AspNetCore.Mvc;
using Farakonesh.API.Filters;
using Farakonesh.Commands.Services;
using Farakonesh.Logic.IDatabase.IOrder;
using Farakonesh.Logic.IServices.IApp.IOrder;
using Farakonesh.Models.Database.StoredProcedures.App.Order;
using Farakonesh.Models.Database.StoredProcedures.App.Order.Payeer;
using System.Threading;
using System.Threading.Tasks;

namespace Farakonesh.API.Controllers.App.Order.Payeer
{
    /// <summary>
    /// سرویس های کار با سفارش شارژ حساب پایر
    /// </summary>
    [Route("api/[controller]/[action]")]
    [ApiController]
    public class OrderPayerChargeController : ControllerBase
    {
        private readonly IPayeerContext _payeerContext;
        private readonly IPayeerService _payeerService;
        public OrderPayerChargeController(IPayeerService payeerService, IPayeerContext payeerContext)
        {
            _payeerService = payeerService;
            _payeerContext = payeerContext;
        }

        /// <summary>
        /// دریافت اطلاعات سفارش - سمت مشتری
        /// </summary>
        /// <param name="inputs"></param>
        /// <returns></returns>
        [HttpPut]
        [CustomersAuth]
        public async Task<IActionResult> GetAll([FromBody] GetOrder_PayerAccountCharge.Inputs inputs, CancellationToken cancellationToken)
        {
            var rst = await _payeerContext.GetOrder_PayerAccountCharge(inputs, cancellationToken);
            return Ok(rst);
        }
        /// <summary>
        /// دریافت اطلاعات سفارش - سمت ادمین
        /// </summary>
        /// <param name="inputs"></param>
        /// <returns></returns>
        [HttpPut]
        public async Task<IActionResult> GetAllByAdmin([FromBody] GetOrder_PayerAccountChargeByAdmin.Inputs inputs, CancellationToken cancellationToken)
        {
            var rst = await _payeerContext.GetOrder_PayerAccountChargeByAdmin(inputs, cancellationToken);
            return Ok(rst);
        }
        /// <summary>
        /// دریافت مبلغ سفارش
        /// </summary>
        /// <param name="inputs"></param>
        /// <returns></returns>
        [HttpPut]
        [CustomersAuth]
        public async Task<IActionResult> GetPrice([FromBody] Get_OrderPayerAccountCharge_Price.Inputs inputs, CancellationToken cancellationToken)
        {
            var rst = await _payeerContext.Get_OrderPayerAccountCharge_Price(inputs, cancellationToken);
            return Ok(rst);
        }
        /// <summary>
        /// ثبت سفارش نوع شارژ حساب پایر
        /// </summary>
        /// <param name="inputs"></param>
        /// <returns></returns>
        [HttpPost]
        [CustomersAuth]
        public async Task<IActionResult> Save([FromBody] Insert_OrderPayerAccountChargeByCustomer.Inputs inputs, CancellationToken cancellationToken)
        {
            await _payeerService.checkAccountBalance(inputs.Amount, inputs.CurrencyId, cancellationToken);

            var rst = await _payeerContext.Insert_OrderPayerAccountChargeByCustomer(inputs, cancellationToken);
            return Ok(rst);
        }
        /// <summary>
        /// ثبت سفارش نوع شارژ حساب پایر - سمت ادمین
        /// </summary>
        /// <param name="inputs"></param>
        /// <returns></returns>
        [HttpPost]
        public async Task<IActionResult> SaveByAdmin([FromBody] Insert_OrderPayerAccountChargeByAdmin.Inputs inputs, CancellationToken cancellationToken)
        {
            await _payeerService.checkAccountBalance(inputs.Amount, inputs.CurrencyId, cancellationToken);

            var rst = await _payeerContext.Insert_OrderPayerAccountChargeByAdmin(inputs, cancellationToken);
            return Ok(rst);
        }
        /// <summary>
        /// ثبت سفارش شارژ حساب پایر - سمت کمپانی
        /// </summary>
        /// <param name="inputs"></param>
        /// <returns></returns>
        [HttpPost]
        [CompanyAuth]
        public async Task<IActionResult> SaveByCompany([FromBody] Insert_OrderPayerAccountChargeByCompany.Inputs inputs, CancellationToken cancellationToken)
        {
            await _payeerService.checkAccountBalance(inputs.Amount, inputs.CurrencyId, cancellationToken);

            var rst = await _payeerContext.Insert_OrderPayerAccountChargeByCompany(inputs, cancellationToken);
            return Ok(rst);
        }
        /// <summary>
        /// انتقال پایر به حساب مشتری - توسط راهبر سیستم
        /// </summary>
        /// <param name="command"></param>
        /// <returns></returns>
        [HttpPost]
        public async Task<IActionResult> TransferByAdmin([FromBody] TransferPayeerCommand command, CancellationToken cancellationToken)
        {
            await _payeerService.checkAccountBalance(command.Amount, command.CurrencyId, cancellationToken);

            var rst = await _payeerService.transferPayeer(command, cancellationToken);
            return Ok(rst);
        }
    }
}
