﻿using Microsoft.AspNetCore.Mvc;
using Farakonesh.API.Filters;
using System.Threading.Tasks;
using Farakonesh.Logic.IDatabase.IUser;
using Farakonesh.Models.Database.StoredProcedures.App.User.User;
using Farakonesh.Commands.Services;
using Farakonesh.Logic.IServices.IApp.IUser;
using System.Threading;

namespace Farakonesh.API.Controllers.App.User
{
    /// <summary>
    /// سرویس های کار با حساب بانکی کاربران
    /// </summary>
    [Route("api/[controller]/[action]")]
    [ApiController]
    public class BankAccountController : ControllerBase
    {
        private readonly IBankAccountContext _ctx;
        private readonly IBankAccountService _bankAccountService;
        public BankAccountController(
    IBankAccountContext ctx,
    IBankAccountService bankAccountService)
        {
            _ctx = ctx;
            _bankAccountService = bankAccountService;
        }
        /// <summary>
        /// دریافت لیست حساب بانکی توسط خود کاربر - سمت مشتری
        /// </summary>
        /// <param name="inputs"></param>
        /// <returns></returns>
        [HttpPut]
        [CustomersAuth]
        public async Task<IActionResult> Get([FromBody] GetBankAccount.Inputs inputs, CancellationToken cancellationToken)
        {
            var rst = await _ctx.GetBankAccount(inputs, cancellationToken);
            return Ok(rst);
        }
        /// <summary>
        /// دریافت لیست حساب بانکی برای یک کاربر توسط اپراتور - سمت ادمین
        /// </summary>
        /// <param name="inputs"></param>
        /// <returns></returns>
        [HttpPut]
        public async Task<IActionResult> GetByAdmin([FromBody] GetBankAccountByAdmin.Inputs inputs, CancellationToken cancellationToken)
        {
            var rst = await _ctx.GetBankAccountByAdmin(inputs, cancellationToken);
            return Ok(rst);
        }
        /// <summary>
        /// افزودن یک حساب بانکی جدید- سمت مشتری
        /// </summary>
        /// <param name="command"></param>
        /// <returns></returns>
        [HttpPost]
        [CustomersAuth]
        public async Task<IActionResult> AddBankAccount([FromBody] BankAccountCommand command, CancellationToken cancellationToken)
        {
            var rst = await _bankAccountService.ActionBankAccount(command, cancellationToken, false);
            return Ok(rst);
        }
        /// <summary>
        /// افزودن یک حساب بانکی جدید برای مشتری - سمت ادمین
        /// </summary>
        /// <param name="command"></param>
        /// <returns></returns>
        [HttpPost]
        public async Task<IActionResult> AddBankAccountByAdmin([FromBody] BankAccountByAdminCommand command, CancellationToken cancellationToken)
        {
            var rst = await _bankAccountService.actionBankAccountByAdmin(command,cancellationToken, false);
            return Ok(rst);
        }
        /// <summary>
        /// بررسی اعتبار حساب بانکی توسط اپراتور - سمت ادمین
        /// </summary>
        /// <param name="command"></param>
        /// <returns></returns>
        [HttpPut]
        public async Task<IActionResult> CheckBankAccountByAdmin([FromBody] CheckBankAccountByAdminCommand command, CancellationToken cancellationToken)
        {
            var rst = await _bankAccountService.checkBankAccountByAdmin(command, cancellationToken);
            return Ok(rst);
        }
        /// <summary>
        /// انتخاب حساب بانکی پیش فرض - سمت مشتری
        /// </summary>
        /// <param name="inputs"></param>
        /// <returns></returns>
        [HttpPut]
        [CustomersAuth]
        public async Task<IActionResult> SetDefault([FromBody] Update_BankAccount_Set_Default.Inputs inputs, CancellationToken cancellationToken)
        {
            var rst = await _ctx.Update_BankAccount_Set_Default(inputs, cancellationToken);
            return Ok(rst);
        }
        /// <summary>
        /// انتخاب حساب بانکی پیش فرض برای مشتری - سمت ادمین
        /// </summary>
        /// <param name="inputs"></param>
        /// <returns></returns>
        [HttpPut]
        public async Task<IActionResult> SetDefaultByAdmin([FromBody] Update_BankAccount_Set_Default_ByAdmin.Inputs inputs, CancellationToken cancellationToken)
        {
            var rst = await _ctx.Update_BankAccount_Set_Default_ByAdmin(inputs, cancellationToken);
            return Ok(rst);
        }
        /// <summary>
        /// حذف حساب بانکی - سمت مشتری
        /// </summary>
        /// <param name="inputs"></param>
        /// <returns></returns>
        [HttpDelete]
        [CustomersAuth]
        public async Task<IActionResult> DeleteBankAccount([FromQuery] Delete_BankAccount.Inputs inputs, CancellationToken cancellationToken)
        {
            var rst = await _ctx.Delete_BankAccount(inputs, cancellationToken);
            return Ok(rst);
        }
        /// <summary>
        /// حذف حساب بانکی - سمت ادمین
        /// </summary>
        /// <param name="inputs"></param>
        /// <returns></returns>
        [HttpDelete]
        public async Task<IActionResult> DeleteBankAccountByAdmin([FromQuery] Delete_BankAccountByAdmin.Inputs inputs, CancellationToken cancellationToken)
        {
            var rst = await _ctx.Delete_BankAccountByAdmin(inputs, cancellationToken);
            return Ok(rst);
        }
    }
}
