﻿using Microsoft.AspNetCore.Authentication.JwtBearer;
using Microsoft.AspNetCore.Hosting;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.IdentityModel.Tokens;
using NSwag.Generation.Processors.Security;
using Farakonesh.Logic.CommonOperations;
using Farakonesh.Logic.Database;
using Farakonesh.Logic.Database.dbo;
using Farakonesh.Logic.Database.Order;
using Farakonesh.Logic.Database.User;
using Farakonesh.Logic.ICommonOperations;
using Farakonesh.Logic.IDatabase.idbo;
using Farakonesh.Logic.IDatabase.IOrder;
using Farakonesh.Logic.IDatabase.IQueue;
using Farakonesh.Logic.IDatabase.IReport;
using Farakonesh.Logic.IDatabase.IUser;
using Farakonesh.Logic.IServices.IApp.IUser;
using Farakonesh.Logic.Services.App;
using Farakonesh.Logic.Services.Cache;
using System.IO;
using System.Text;
using System.Threading.Tasks;
using System.Reflection;

namespace Farakonesh.API.Extensions
{
    public static class Services
    {
        public const string corsPolicyName = "_AllowOrigin";

        /// <summary>
        /// اعمال دسترسی به دامنه های بیرونی
        /// </summary>
        /// <param name="services"></param>
        /// <param name="configuration"></param>
        public static void ConfigureCors(this IServiceCollection services, IConfiguration configuration)
        {
            services.AddCors(options =>
            {
                options.AddPolicy(
                        corsPolicyName,
                        builder =>
                            builder
                           .WithOrigins(configuration.GetValue<string>("AllowedOrigin").Split(","))
                            //صدور مجوز اجرا از هر دامنه ای
                            // .AllowAnyOrigin()
                            //صدور مجوز اجرا تمامی انواع متدها
                            .AllowAnyMethod()
                            //صدور مجوز اجرا با هر مقدار هدری
                            .AllowAnyHeader()
                            );
            });

        }


        /// <summary>
        /// Setup JWT validator
        /// </summary>
        /// <param name="services"></param>
        /// <param name="configuration"></param>
        public static void AddJWT(this IServiceCollection services, IConfiguration configuration)
        {
            string key = configuration.GetValue<string>("JWT:Key");
            string issuer = configuration.GetValue<string>("JWT:Issuer");

            services
                .AddAuthentication(JwtBearerDefaults.AuthenticationScheme)
                .AddJwtBearer(options =>
                {
                    options.TokenValidationParameters = new TokenValidationParameters
                    {
                        ValidateIssuer = true,
                        ValidateAudience = true,
                        ValidateIssuerSigningKey = true,
                        ValidIssuer = issuer,
                        ValidAudience = issuer,
                        IssuerSigningKey = new SymmetricSecurityKey(Encoding.UTF8.GetBytes(key))
                    };
                    options.Events = new JwtBearerEvents
                    {
                        OnAuthenticationFailed = context =>
                        {
                            if (context.Exception.GetType() == typeof(SecurityTokenExpiredException))
                                context.Response.Headers.Add("Token-Expired", "true");

                            return Task.CompletedTask;
                        }
                    };
                });
        }


        /// <summary>
        /// Set swagger setttings
        /// </summary>
        /// <param name="services"></param>
        public static void SetSwaggerSettings(this IServiceCollection services)
        {
            services.AddSwaggerDocument(c =>
            {
                c.DocumentName = "Farakonesh, "+ GetAppVersion();
                c.Title = "Farakonesh API Document";
                c.Version = GetAppVersion();
                c.Description = "Farakonesh API {GET,Post,Put,Delete,Patch}";
                c.OperationProcessors.Add(new OperationSecurityScopeProcessor("Bearer"));
                c.GenerateExamples = true;
                c.GenerateCustomNullableProperties = true;
                c.AllowNullableBodyParameters = true;
                c.AddSecurity("Bearer", new NSwag.OpenApiSecurityScheme()
                {
                    Description = "Authorization header using the Bearer scheme. Example: \"Authorization: Bearer {token}\"",
                    Name = "Authorization",
                    In = NSwag.OpenApiSecurityApiKeyLocation.Header,
                    Type = NSwag.OpenApiSecuritySchemeType.ApiKey,
                    BearerFormat = "Authorization: Bearer {token}"
                });
                c.PostProcess = document =>
                {
                    document.Info.TermsOfService = "none";
                    document.Info.Contact = new NSwag.OpenApiContact
                    {
                        Name = "Farakonesh",
                        Email = "info@farakonesh.org",
                        Url = "farakonesh.org"
                    };
                };
            });
        }

        public static string GetAppVersion()
        {
            var version = Assembly
                .GetExecutingAssembly()
                .GetCustomAttribute<AssemblyInformationalVersionAttribute>()?
                .InformationalVersion;

            return version ?? "1.0.0";
        }
    }
}
