﻿using Microsoft.AspNetCore.Http;
using System.Text;
using System.Threading.Tasks;
using System;
using Microsoft.Extensions.Options;
using System.Collections.Generic;
using Farakonesh.Models.SystemSetting;
using Farakonesh.Logic.IServices.ICache;
using Farakonesh.Logic.Services.Cache;
using System.Threading;

namespace Farakonesh.API.Middlewares
{
    public class BasicAuthSwaggerMiddleware
    {
        private readonly RedisService _redisService;
        private readonly RequestDelegate _next;
        private readonly SwaggerAuthSettings _authSettings;
        private static readonly TimeSpan BlockDuration = TimeSpan.FromMinutes(10);
        private const int MaxFailedAttempts = 5;

        public BasicAuthSwaggerMiddleware(RequestDelegate next, IOptions<SwaggerAuthSettings> authSettings,
            RedisService redisService)
        {
            _next = next;
            _authSettings = authSettings.Value;
            _redisService = redisService;
        }

        public async Task Invoke(HttpContext context)
        {
            if (context.Request.Path.StartsWithSegments("/swagger"))
            {
                var ip = context.Connection.RemoteIpAddress?.ToString() ?? "unknown";

                if (await IsBlockedAsync(ip))
                {
                    context.Response.StatusCode = 403;
                    context.Response.ContentType = "text/plain; charset=utf-8";
                    await context.Response.WriteAsync("تعداد تلاش ناموفق شما از حد مجاز فراتر رفته است ، لطفا بعدا مجددا تلاش کنید");
                    return;
                }

                if (!context.Request.Headers.ContainsKey("Authorization"))
                {
                    await Challenge(context);
                    await RegisterFailedAttemptAsync(ip);
                    return;
                }

                var authHeader = context.Request.Headers["Authorization"].ToString();
                if (authHeader.StartsWith("Basic ", StringComparison.OrdinalIgnoreCase))
                {
                    var token = authHeader.Substring("Basic ".Length).Trim();
                    var credentialString = Encoding.UTF8.GetString(Convert.FromBase64String(token));
                    var credentials = credentialString.Split(':');

                    if (credentials.Length == 2 &&
                        credentials[0] == _authSettings.Username &&
                        credentials[1] == _authSettings.Password)
                    {
                        await ResetFailures(ip);
                        await _next(context);
                        return;
                    }
                }

                await _redisService.Remove(ip);
                await Challenge(context);
                return;
            }

            await _next(context);
        }

        private async Task RegisterFailedAttemptAsync(string ip)
        {
            var info = await _redisService.Get<FailedLoginInfo>(ip);
            if (info == null)
            {
                info = new FailedLoginInfo();
            }

            info.FailedCount++;
            info.LastFailedTime = DateTime.UtcNow;

            if (info.FailedCount >= MaxFailedAttempts)
            {
                info.BlockCount++;

                if (info.BlockCount >= 3)
                {
                    info.IsPermanentlyBlocked = true;
                }
            }

            await _redisService.Set<FailedLoginInfo>(ip, info);
        }



        private async Task<bool> IsBlockedAsync(string ip)
        {
            var info = await _redisService.Get<FailedLoginInfo>(ip);
            if (info == null)
                return false;

            if (info.IsPermanentlyBlocked)
                return true;

            if (info.FailedCount >= MaxFailedAttempts &&
                DateTime.UtcNow - info.LastFailedTime < BlockDuration)
                return true;

            return false;
        }


        private async Task ResetFailures(string ip)
        {
            await _redisService.Remove(ip);
        }

        private Task Challenge(HttpContext context)
        {
            context.Response.StatusCode = 401;
            context.Response.Headers["WWW-Authenticate"] = "Basic realm=\"Swagger UI\"";
            return context.Response.WriteAsync("Unauthorized");
        }

    }
}
