﻿using Microsoft.AspNetCore.Http;
using Microsoft.Extensions.Logging;
using NLog;
using Farakonesh.Logic.IDatabase;
using Farakonesh.Logic.Log;
using StackExchange.Redis;
using System.IO;
using System.Text;
using System.Threading.Tasks;
using System.IO.Compression;

namespace Farakonesh.API.Middlewares
{
    public class LoggingMiddleware
    {
        private readonly RequestDelegate _next;
        private readonly ILog _logger;

        public LoggingMiddleware(RequestDelegate next,ILog logger)
        {
            _next = next;
            _logger = logger;
        }

        public async Task Invoke(HttpContext context)
        {
            logRequest(context);

            var originalBodyStream = context.Response.Body;
            using (var responseBody = new MemoryStream())
            {
                context.Response.Body = responseBody;

                await _next(context);

                await logResponse(context, responseBody);

                responseBody.Seek(0, SeekOrigin.Begin);
                await responseBody.CopyToAsync(originalBodyStream);
            }
        }


        private void logRequest(HttpContext context)
        {
            var request = context.Request;

            var requestLog = new StringBuilder();
            requestLog.AppendLine("Incoming Request:");
            requestLog.AppendLine($"HTTP {request.Method} {request.Path}");
            requestLog.AppendLine($"Host: {request.Host}");
            requestLog.AppendLine($"Content-Type: {request.ContentType}");
            requestLog.AppendLine($"Content-Length: {request.ContentLength}");

            _logger.Information(requestLog.ToString(), 3);
        }

        private async Task logResponse(HttpContext context, MemoryStream responseBody)
        {
            responseBody.Seek(0, SeekOrigin.Begin);
            string text = await new StreamReader(responseBody).ReadToEndAsync();

            var response = context.Response;

            var responseLog = new StringBuilder();
            responseLog.AppendLine("Outgoing Response:");
            responseLog.AppendLine($"HTTP {response.StatusCode}");
            responseLog.AppendLine($"Content-Type: {response.ContentType}");
            responseLog.AppendLine($"Content-Length: {response.ContentLength}");
            responseLog.AppendLine("Response Body:");
            responseLog.AppendLine(text);

            _logger.Information(responseLog.ToString(), 3);
        }

      
    }
}
