﻿using System;
using System.Data;
using System.IO.Compression;
using System.Linq;
using AspNetCoreRateLimit;
using Farakonesh.API.Extensions;
using Farakonesh.API.Filters;
using Farakonesh.API.Middlewares;
using Farakonesh.IOC;
using Farakonesh.Logic.Database.dbo;
using Farakonesh.Logic.IDatabase.idbo;
using Farakonesh.Logic.Log;
using Farakonesh.Logic.Services.Cache;
using Farakonesh.Models.SystemSetting;
using Farakonesh.Shared.Filters;
using Microsoft.AspNetCore.Builder;
using Microsoft.AspNetCore.Diagnostics.HealthChecks;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.HttpOverrides;
using Microsoft.AspNetCore.ResponseCompression;
using Microsoft.AspNetCore.Server.Kestrel.Core;
using Microsoft.Data.SqlClient;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.Extensions.Diagnostics.HealthChecks;
using Microsoft.Extensions.Hosting;
using Microsoft.Extensions.Options;
using Newtonsoft.Json;
using NLog;
using NLog.Web;

var logger = NLogBuilder.ConfigureNLog("nlog.config").GetCurrentClassLogger();

try
{
    var builder = WebApplication.CreateBuilder(args);

    builder.Host.UseNLog();

    var configuration = builder.Configuration;
    var services = builder.Services;

    services.Configure<KestrelServerOptions>(options => options.AllowSynchronousIO = false);
    services.Configure<IISServerOptions>(options => options.AllowSynchronousIO = false);

    services.AddSingleton<ILog, Log>();
    services.AddSingleton<IConfiguration>(configuration);
    services.AddHttpClient();
    services.AddHttpContextAccessor();
    services.AddScoped<ISettingServerContext, SettingServerContext>();
    services.AddSingleton<RedisService>();
    services.AddHostedService<RedisInitializerHostedService>();
    services.AddHealthChecks()
     .AddSqlServer(
         connectionString: configuration.GetConnectionString("cns"),
         healthQuery: "SELECT 1;",
         name: "sql",
         failureStatus: HealthStatus.Unhealthy)
     .AddRabbitMQ(configuration.GetConnectionString("RabbitMQConnection"), name: "rabbitmq")
     .AddRedis(configuration.GetConnectionString("RedisCns"), name: "redis")
     .AddWorkingSetHealthCheck(1_073_741_824, name: "memory")
     .AddDiskStorageHealthCheck(options =>
     {
         options.AddDrive("C:\\", minimumFreeMegabytes: 500);
     }, name: "disk");
    services.AddJWT(configuration);

    services.Configure<SwaggerAuthSettings>(configuration.GetSection("SwaggerAuthSettings"));
    services.Configure<MiddlewareSettings>(configuration.GetSection("MiddlewareSettings"));

    services.BuildContainer();

    services.SetSwaggerSettings();

    services.AddResponseCompression(option =>
    {
        option.EnableForHttps = true;
        option.Providers.Add<GzipCompressionProvider>();
    });
    services.Configure<GzipCompressionProviderOptions>(options => options.Level = CompressionLevel.Fastest);
    services.Configure<IpRateLimitOptions>(builder.Configuration.GetSection("IpRateLimiting"));
    services.Configure<ClientRateLimitOptions>(builder.Configuration.GetSection("ClientRateLimiting"));
    services.AddSingleton<IRateLimitConfiguration, RateLimitConfiguration>();
    services.AddSingleton<IClientPolicyStore, MemoryCacheClientPolicyStore>();
    services.AddSingleton<IIpPolicyStore, MemoryCacheIpPolicyStore>();
    services.AddSingleton<IRateLimitCounterStore, MemoryCacheRateLimitCounterStore>();
    services.AddSingleton<IProcessingStrategy, AsyncKeyLockProcessingStrategy>();
    services.AddMemoryCache();
    services.AddOptions();
    services.AddControllers(options =>
    {
        options.Filters.Add<SanitizeHtmlFilter>();
    })
        .AddNewtonsoftJson(options =>
        {
            options.SerializerSettings.NullValueHandling = Newtonsoft.Json.NullValueHandling.Include;
            options.SerializerSettings.DefaultValueHandling = Newtonsoft.Json.DefaultValueHandling.Include;
            options.SerializerSettings.MissingMemberHandling = Newtonsoft.Json.MissingMemberHandling.Ignore;
            options.SerializerSettings.Formatting = Newtonsoft.Json.Formatting.Indented;
            options.SerializerSettings.ContractResolver = new Newtonsoft.Json.Serialization.CamelCasePropertyNamesContractResolver();
        });
    services.AddScoped<SanitizeHtmlFilter>();
    services.AddMvc(config =>
    {
        config.Filters.Add(new AuthAttribute());
    });

    services.ConfigureCors(configuration);

    var app = builder.Build();

    var env = app.Environment;
    var swaggerOptions = app.Services.GetRequiredService<IOptions<SwaggerAuthSettings>>();
    var middlewareSettings = app.Services.GetRequiredService<IOptions<MiddlewareSettings>>();

    if (env.IsDevelopment())
    {
        app.UseDeveloperExceptionPage();
    }
    app.UseMiddleware<CorrelationIdMiddleware>();
    app.UseRouting();
    app.UseCors(Services.corsPolicyName);
    app.UseForwardedHeaders(new ForwardedHeadersOptions
    {
        ForwardedHeaders = ForwardedHeaders.XForwardedFor | ForwardedHeaders.XForwardedProto
    });
    app.UseMiddleware<ExceptionMiddleware>();
    if (middlewareSettings.Value.PerformanceMiddleware == true)
    {
        app.UseMiddleware<PerformanceMiddleware>();
    }
    app.UseMiddleware<ModifyQueryStringMiddleware>();
    app.UseMiddleware<BasicAuthSwaggerMiddleware>();
    if (middlewareSettings.Value.LoggingMiddleware == true)
    {
        app.UseMiddleware<LoggingMiddleware>();
    }
    app.UseOpenApi();
    if (swaggerOptions.Value.IsActive)
    {
        app.UseSwaggerUi3();
    }
    app.UseHsts();
    app.UseXContentTypeOptions();
    app.UseReferrerPolicy(opts => opts.NoReferrer());
    app.UseXXssProtection(options => options.EnabledWithBlockMode());
    app.UseXfo(options => options.Deny());
    app.UseCsp(options => options
    .DefaultSources(s => s.Self())
    .ScriptSources(s => s.Self().UnsafeInline())
    .StyleSources(s => s.Self().UnsafeInline())
    .FontSources(s => s.Self())
    .ImageSources(s => s.Self())
    .FrameAncestors(s => s.None()));
    app.UseResponseCompression();
    app.UseHttpsRedirection();
    app.UseAuthentication();
    app.UseAuthorization();
    app.UseDefaultFiles();

    app.UseIpRateLimiting();
    app.UseClientRateLimiting();
    app.MapRazorPages();
    app.MapControllers();

    app.MapGet("/", () =>
    {
        var version = Services.GetAppVersion();
        return Results.Ok(new { version });
    });
    app.MapHealthChecks("/health", new HealthCheckOptions
    {
        ResponseWriter = async (context, report) =>
        {
            var token = context.Request.Query["token"].ToString();
            const string secretToken = "F@rakonesh2025";

            if (string.IsNullOrEmpty(token) || token != secretToken)
            {
                context.Response.StatusCode = StatusCodes.Status401Unauthorized;
                await context.Response.WriteAsync("Unauthorized");
                return;
            }

            context.Response.ContentType = "application/json";
            var result = JsonConvert.SerializeObject(new
            {
                status = report.Status.ToString(),
                checks = report.Entries.Select(e => new
                {
                    name = e.Key,
                    status = e.Value.Status.ToString(),
                    exception = e.Value.Exception?.Message,
                    duration = e.Value.Duration.ToString()
                })
            }, Formatting.Indented);

            await context.Response.WriteAsync(result);
        }
    });
    app.Use(async (context, next) =>
    {
        if (context.User.Identity?.IsAuthenticated == true)
        {
            var clientId = context.User.FindFirst("UserId")?.Value;
            if (!string.IsNullOrWhiteSpace(clientId))
            {
                context.Request.Headers["X-ClientId"] = clientId;
            }
        }

        await next();
    });
    await app.RunAsync();
}
catch (Exception ex)
{
    logger.Error(ex, "Stopped program because of exception");
    throw;
}
finally
{
    LogManager.Shutdown();
}
