﻿using Dapper;
using Farakonesh.Models.Database;
using Farakonesh.Models.Database.StoredProcedures.App.dbo.setting;
using Farakonesh.Models.Database.StoredProcedures.App.User.User;
using Farakonesh.Shared.Exceptions;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Mvc;
using RestSharp;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Data;
using System.Dynamic;
using System.Linq;
using System.Reflection;
using System.Threading;
using System.Threading.Tasks;

namespace Farakonesh.DP.DapperORM
{
    public class BaseSP<TInput, TOutput>
    {

        private readonly string _spName;

        private readonly ISqlExecuter _executer;
        private readonly Common _common;
        public BaseSP(string SPName, ISqlExecuter executer)
        {
            _spName = SPName;
            _executer = executer;
            _common = new Common();

        }
        private static readonly ConcurrentDictionary<Type, PropertyInfo[]> _propertyCache = new ConcurrentDictionary<Type, PropertyInfo[]>();

        private static PropertyInfo[] GetCachedProperties(Type type)
        {
            return _propertyCache.GetOrAdd(type, t => t.GetProperties());
        }

        private DynamicParameters BuildParameters(TInput inputs, bool forPagination = false)
        {
            if (inputs == null)
                throw new RequestException("داده های ورودی نا معتبر می باشد");

            var fields = GetCachedProperties(inputs.GetType());
            var dataParams = forPagination ? _common.getSearchParams() : _common.getEmptyParams();

            foreach (var f in fields)
            {
                var value = f.GetValue(inputs);
                if (value != null)
                {
                    dataParams.Add(f.Name, value);
                }
                else
                {
                    if (f.PropertyType == typeof(byte[]))
                        dataParams.Add(f.Name, DBNull.Value, DbType.Binary);
                    else
                        dataParams.Add(f.Name, null);
                }
            }

            return dataParams;
        }

        private async Task<ResultQueryModel<TOutput>> GetAll(TInput inputs, CancellationToken cancellationToken)
        {
            var dataParams = BuildParameters(inputs);
            var result = await _executer.QueryAsync<TOutput>(cancellationToken,_spName, dataParams, CommandType.StoredProcedure);
            return new ResultQueryModel<TOutput> { Result = result, Parameters = dataParams };
        }

        private async Task<ResultQuerySingleModel<TOutput>> GetMultiple(TInput inputs, CancellationToken cancellationToken)
        {
            var dataParams = BuildParameters(inputs);

            try
            {
                using (var multi = await _executer.QueryMultipleAsync(
                    cancellationToken,
                    _spName,
                    dataParams,
                    CommandType.StoredProcedure))
                {
                    var resultObj = Activator.CreateInstance<TOutput>();

                    foreach (var prop in typeof(TOutput).GetProperties())
                    {
                        var propType = prop.PropertyType;
                        if (propType.IsGenericType && propType.GetGenericTypeDefinition() == typeof(DBResult<>))
                        {
                            var innerType = propType.GetGenericArguments()[0];
                            if (innerType.IsGenericType && innerType.GetGenericTypeDefinition() == typeof(IEnumerable<>))
                            {
                                var itemType = innerType.GetGenericArguments()[0];
                                var list = await multi.ReadAsync(itemType);
                                var castedList = CastList(list, itemType);
                                var container = Activator.CreateInstance(propType);
                                propType.GetProperty("Data").SetValue(container, castedList);
                                prop.SetValue(resultObj, container);
                            }
                            else
                            {
                                var single = (await multi.ReadAsync(innerType)).FirstOrDefault();

                                var container = Activator.CreateInstance(propType);
                                propType.GetProperty("Data").SetValue(container, single);
                                prop.SetValue(resultObj, container);
                            }

                        }
                        
                    }

                    return new ResultQuerySingleModel<TOutput>
                    {
                        Result = resultObj,
                        Parameters = dataParams
                    };
                }
            }
            catch (Exception ex)
            {
                throw new QueryException($"SP: {_spName}, Error: {ex.Message}");
            }
        }
   
        private async Task<ResultQuerySingleModel<TOutput>> GetSingleOrDefault(TInput inputs, CancellationToken cancellationToken)
        {
            var dataParams = BuildParameters(inputs);
            var result = await _executer.FirstOrDefaultAsync<TOutput>(cancellationToken, _spName, dataParams, System.Data.CommandType.StoredProcedure);
            return new ResultQuerySingleModel<TOutput> { Result = result, Parameters = dataParams };
        }
        private async Task<ResultQuerySingleModel<TOutput>> GetFirstOrDefault(TInput inputs, CancellationToken cancellationToken)
        {
            var dataParams = BuildParameters(inputs);
            var result = await _executer.FirstOrDefaultAsync<TOutput>(cancellationToken, _spName, dataParams, System.Data.CommandType.StoredProcedure);
            return new ResultQuerySingleModel<TOutput> { Result = result, Parameters = dataParams };
        }

        private async Task<ResultQueryModel<TOutput>> GetAllPagination(TInput inputs, CancellationToken cancellationToken)
        {
            var dataParams = BuildParameters(inputs,true);
            var result = await _executer.QueryAsync<TOutput>(cancellationToken, _spName, dataParams, System.Data.CommandType.StoredProcedure);
            return new ResultQueryModel<TOutput> { Parameters = dataParams, Result = result };
        }
        public async Task<DBResult> Execute(TInput inputs, CancellationToken cancellationToken)
        {
            var dataParams = BuildParameters(inputs);
            await _executer.ExecuteAsync(cancellationToken, _spName, dataParams, System.Data.CommandType.StoredProcedure);
            return new DBResult();
        }

    

        public async Task<DBResult<TOutput>> Single(TInput input, CancellationToken cancellationToken)
        {
            var result = await GetFirstOrDefault(input,cancellationToken);
            if (result == null)
            {
                throw new QueryException($"خطا در دریافت نتیجه رخ داده است ، درصورت تکرار لطفا با کارشناسان برنامه ارتباط برقرار کنید");
            }
            return Common.GetResult<TOutput>(result.Parameters, result.Result);
        }
    
   
        public async Task<DBResult<IEnumerable<TOutput>>> Query(TInput input, CancellationToken cancellationToken)
        {
            var result = await GetAll(input, cancellationToken);
            return Common.GetResult<IEnumerable<TOutput>>(result.Parameters, result.Result);
        }

        public async Task<DBResult<TOutput>> QueryMultiple(TInput input, CancellationToken cancellationToken)
        {
            var result = await GetMultiple(input, cancellationToken);
            return Common.GetResult<TOutput>(result.Parameters, result.Result);
        }

        public async Task<DBResult> QueryFirstOrDefault(TInput input, CancellationToken cancellationToken)
        {
            var result = await GetFirstOrDefault(input,cancellationToken);
            return Common.GetResult(result.Parameters, (object)result.Result);
        }

   

        public async Task<DBResult<IEnumerable<TOutput>>> QueryPagination(TInput input, CancellationToken cancellationToken)
        {
            var result = await GetAllPagination(input, cancellationToken);
            return Common.GetResult(result.Parameters, result.Result);
        }

        private static object CastList(IEnumerable<object> list, Type targetType)
        {
            var method = typeof(Enumerable).GetMethod("Cast")?.MakeGenericMethod(targetType);
            var casted = method?.Invoke(null, new object[] { list });

            var toListMethod = typeof(Enumerable).GetMethod("ToList")?.MakeGenericMethod(targetType);
            return toListMethod?.Invoke(null, new object[] { casted });
        }

    }

}