﻿using Microsoft.Extensions.Configuration;
using Newtonsoft.Json;
using RabbitMQ.Client;
using RestSharp;
using Farakonesh.Logic.ICommonOperations;
using Farakonesh.Logic.IDatabase;
using Farakonesh.Logic.IDatabase.idbo;
using Farakonesh.Logic.IDatabase.IQueue;
using Farakonesh.Logic.IQueue;
using Farakonesh.Logic.IServices.ICache;
using Farakonesh.Logic.IServices.IRestRequest;
using Farakonesh.Logic.Services.Cache;
using Farakonesh.Logic.Services.RestRequest;
using Farakonesh.Models.API.Email;
using Farakonesh.Models.API.Pakat;
using Farakonesh.Models.API.RabbitMQ;
using Farakonesh.Models.Database.StoredProcedures.App;
using Farakonesh.Models.Database.StoredProcedures.App.dbo.setting;
using Farakonesh.Models.Database.StoredProcedures.App.Queue;
using Farakonesh.Models.Database.StoredProcedures.App.User.Authentication;
using StackExchange.Redis;
using System;
using System.Collections.Generic;
using System.Data.SqlTypes;
using System.IO;
using System.Net;
using System.Net.Http;
using System.Net.Mail;
using System.Net.Security;
using System.Net.Sockets;
using System.Security.Authentication;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Farakonesh.Shared.Helpers;
using Microsoft.AspNetCore.Http;
using Farakonesh.Shared.Enums;

namespace Farakonesh.Logic.CommonOperations
{
    public class MailHelper : IMailHelper
    {
        private readonly IQueueContext _IQueueContext;
        private readonly IHttpRequestEmail _httpRequestEmail;
        private readonly ISendEmailQueue _sendEmailQueue;
        private readonly IRedisContextService _redisContextService;
        private readonly IHttpClientFactory _httpClientFactory;
        private readonly IHttpContextAccessor _httpContextAccessor;
        public MailHelper(IQueueContext iQueueContext,
            IHttpRequestEmail httpRequestEmail, ISendEmailQueue sendEmailQueue
            , IRedisContextService redisContextService, IHttpClientFactory httpClientFactory,
            IHttpContextAccessor httpContextAccessor)
        {
            _httpContextAccessor = httpContextAccessor;
            _IQueueContext = iQueueContext;
            _httpRequestEmail = httpRequestEmail;
            _sendEmailQueue = sendEmailQueue;
            _redisContextService = redisContextService;
            _httpClientFactory = httpClientFactory;
        }
        /// <summary>
        /// ارسال ایمیل با قالب ایمیل اچ تی ام ال
        /// </summary>
        /// <param name="model"></param>
        /// <returns></returns>
        public async Task<string> sendWithTemplatePath(EmailRequestSendModel model, CancellationToken cancellationToken)
        {
            if (string.IsNullOrWhiteSpace(model.Email))
            {
                return string.Empty;
            }
            var setting = await _redisContextService.GetSettingEmailAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.TransactionalEmailHostiran,
                cancellationToken);
            var config = webServiceInfo.AdditionalConfig.JsonParse();
            var localIp = _httpContextAccessor.HttpContext?.Connection.LocalIpAddress;
            var isExternalServer = await IpCheckHelper.IsExternalServerAsync(setting.ExternalServerIp, new IpifyPublicIpProvider()
, localIp);
            if (((isExternalServer && setting.IsActiveQueueExternalServer == true)
                || (!isExternalServer && setting.IsActiveQueueInternalServer == true)) && setting.SendEmailFromQueue == true)
            {
                await _sendEmailQueue.sendToQueueTransAction(new EmailSendModel
                {
                    email = model.Email,
                    text = model.Body,
                    title = model.Subject
                });
                return string.Empty;
            }

            if (setting.IsActiveApiMail == true)
            {
                return await _httpRequestEmail.sendEmail(setting.BaseUrlApiMail, model, cancellationToken);
            }

            var dbResultEmail = await _IQueueContext.Insert_Email(new Insert_Email.Inputs { Email = model.Email, Text = model.Body }, cancellationToken);

            string textResponse = null;
            try
            {
                using (MailMessage mail = new MailMessage())
                using (SmtpClient SmtpServer = new SmtpClient((string)config["server"]))
                {
                    mail.From = new MailAddress((string)config["sender"], setting.SystemTitle);
                    mail.To.Add(new MailAddress(model.Email));
                    mail.Subject = model.Subject;
                    mail.SubjectEncoding = Encoding.UTF8;
                    var client = _httpClientFactory.CreateClient();
                    string mailText = await client.GetStringAsync((string)config["templateAddress"]);

                    mailText = mailText.Replace("{{siteurl}}", setting.SiteUrlRoot).Replace("{{sitelogo}}"
                        , setting.LogoAddress).Replace("{{sitesubject}}", model.Subject).Replace("{{sitedescription}}", model.Body);


                    mail.Body = mailText;

                    mail.IsBodyHtml = true;

                    SmtpServer.DeliveryMethod = SmtpDeliveryMethod.Network;

                    SmtpServer.Port = Convert.ToInt32((string)config["port"]);

                    SmtpServer.UseDefaultCredentials = false;

                    SmtpServer.Credentials = new NetworkCredential(webServiceInfo.Username, webServiceInfo.RealPassword);

                    SmtpServer.EnableSsl = true;

                    await SmtpServer.SendMailAsync(mail);
                }
            }
            catch (SmtpFailedRecipientException ex)
            {
                textResponse = ex.Message;
            }
            catch (SmtpException ex)
            {
                textResponse = ex.Message;
            }
            catch (Exception ex)
            {
                textResponse = ex.Message;
            }
            bool isDone = false;
            if (string.IsNullOrWhiteSpace(textResponse))
                isDone = true;
            await _IQueueContext.Update_Email(new Update_Email.Inputs
            {
                IsDone = isDone,
                EmailId = Guid.Parse(dbResultEmail.Data.getValue("EmailId").ToString()),
                Text = !string.IsNullOrWhiteSpace(textResponse) ? textResponse : null
            }, cancellationToken);
            return textResponse;
        }

        public async Task<string> checkConnection(CancellationToken cancellationToken)
        {
            bool enableSsl = false;

            var setting = await _redisContextService.GetSettingEmailAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.TransactionalEmailHostiran,
              cancellationToken);
            var config = webServiceInfo.AdditionalConfig.JsonParse();
            if (setting.IsActiveApiMail == true)
            {
                return await _httpRequestEmail.checkEmail(setting.BaseUrlApiMail, cancellationToken);
            }

            try
            {
                using (var client = new TcpClient())
                {
                    var server = (string)config["server"];
                    var port = Convert.ToInt32((string)config["port"]);

                    SmtpConnectorBase connector;
                    if (enableSsl)
                    {
                        connector = new SmtpConnectorWithSsl(server, port);
                    }
                    else
                    {
                        connector = new SmtpConnectorWithoutSsl(server, port);
                    }

                    if (!await connector.CheckResponse(220, cancellationToken))
                    {
                        return "error-" + $"هشدار : ارتباط اولیه با سرور ایمیل برقرار نشد ، لطفا جهت جلوگیری از هرگونه اختلال در کارکرد سیستم این مورد را بررسی کنید";
                    }

                    connector.SendData($"HELO {Dns.GetHostName()}{SmtpConnectorBase.EOF}");
                    if (!await connector.CheckResponse(250, cancellationToken))
                    {
                        return "error-" + $"هشدار : ارتباط با سرور ایمیل برقرار نشد ، لطفا جهت جلوگیری از هرگونه اختلال در کارکرد سیستم این مورد را بررسی کنید";
                    }

                    connector.SendData($"AUTH LOGIN{SmtpConnectorBase.EOF}");
                    if (!await connector.CheckResponse(334, cancellationToken))
                    {
                        return "error-" + $"هشدار : احراز هویت در سرور ایمیل معتبر نبود ، لطفا جهت جلوگیری از هرگونه اختلال در کارکرد سیستم این مورد را بررسی کنید";
                    }

                    connector.SendData(Convert.ToBase64String(Encoding.UTF8.GetBytes($"{webServiceInfo.Username}")) + SmtpConnectorBase.EOF);
                    if (!await connector.CheckResponse(334, cancellationToken))
                    {
                        return "error-" + $"هشدار : احراز هویت در سرور ایمیل معتبر نبود ، لطفا جهت جلوگیری از هرگونه اختلال در کارکرد سیستم این مورد را بررسی کنید";
                    }

                    connector.SendData(Convert.ToBase64String(Encoding.UTF8.GetBytes($"{webServiceInfo.RealPassword}")) + SmtpConnectorBase.EOF);
                    if (!await connector.CheckResponse(235, cancellationToken))
                    {
                        return "error-" + $"هشدار : احراز هویت در سرور ایمیل معتبر نبود ، لطفا جهت جلوگیری از هرگونه اختلال در کارکرد سیستم این مورد را بررسی کنید";
                    }

                    return "success-" + $"ارتباط با سرویس دهنده ایمیل پایدار و معتبر است";

                }
            }
            catch (Exception e)
            {
                return "error-" + $"هشدار : ارتباط با سرور ایمیل برقرار نشد ، لطفا جهت جلوگیری از هرگونه اختلال در کارکرد سیستم این مورد را بررسی کنید";
            }

        }
    }

    internal abstract class SmtpConnectorBase
    {
        protected string SmtpServerAddress { get; set; }
        protected int Port { get; set; }
        public const string EOF = "\r\n";

        protected SmtpConnectorBase(string smtpServerAddress, int port)
        {
            SmtpServerAddress = smtpServerAddress;
            Port = port;
        }

        public abstract Task<bool> CheckResponse(int expectedCode,CancellationToken cancellationToken);
        public abstract void SendData(string data);
    }
    internal class SmtpConnectorWithoutSsl : SmtpConnectorBase
    {
        private Socket _socket = null;

        public SmtpConnectorWithoutSsl(string smtpServerAddress, int port) : base(smtpServerAddress, port)
        {
            IPHostEntry hostEntry = Dns.GetHostEntry(smtpServerAddress);
            IPEndPoint endPoint = new IPEndPoint(hostEntry.AddressList[0], port);
            _socket = new Socket(endPoint.AddressFamily, SocketType.Stream, ProtocolType.Tcp);
            _socket.Connect(endPoint);

        }

        ~SmtpConnectorWithoutSsl()
        {
            try
            {
                if (_socket != null)
                {
                    _socket.Close();
                    _socket.Dispose();
                    _socket = null;
                }
            }
            catch (Exception)
            {
                ;
            }

        }

        public override async Task<bool> CheckResponse(int expectedCode,CancellationToken cancellationToken)
        {
            while (_socket.Available == 0)
            {
                Thread.Sleep(100);
            }
            byte[] responseArray = new byte[1024];
            await _socket.ReceiveAsync(new ArraySegment<byte>(responseArray), SocketFlags.None, cancellationToken);
            string responseData = Encoding.UTF8.GetString(responseArray);
            int responseCode = Convert.ToInt32(responseData.Substring(0, 3));
            if (responseCode == expectedCode)
            {
                return true;
            }
            return false;
        }

        public override void SendData(string data)
        {
            byte[] dataArray = Encoding.UTF8.GetBytes(data);
            _socket.Send(dataArray, 0, dataArray.Length, SocketFlags.None);
        }
    }

    internal class SmtpConnectorWithSsl : SmtpConnectorBase
    {
        private SslStream _sslStream = null;
        private TcpClient _client = null;

        public SmtpConnectorWithSsl(string smtpServerAddress, int port) : base(smtpServerAddress, port)
        {
            TcpClient client = new TcpClient(smtpServerAddress, port);

            _sslStream = new SslStream(
                client.GetStream(),
                false,
                new RemoteCertificateValidationCallback(ValidateServerCertificate),
                null
                );

            try
            {
                _sslStream.AuthenticateAsClient(smtpServerAddress);
            }
            catch (AuthenticationException e)
            {
                _sslStream = null;
                Console.WriteLine("Exception: {0}", e.Message);
                if (e.InnerException != null)
                {
                    Console.WriteLine("Inner exception: {0}", e.InnerException.Message);
                }
                Console.WriteLine("Authentication failed - closing the connection.");
                client.Close();
            }
        }

        ~SmtpConnectorWithSsl()
        {
            try
            {
                if (_sslStream != null)
                {
                    _sslStream.Close();
                    _sslStream.Dispose();
                    _sslStream = null;
                }
            }
            catch (Exception)
            {
                ;
            }

            try
            {
                if (_client != null)
                {
                    _client.Close();
                    _client = null;
                }
            }
            catch (Exception)
            {
                ;
            }
        }

        private static bool ValidateServerCertificate(
              object sender,
              X509Certificate certificate,
              X509Chain chain,
              SslPolicyErrors sslPolicyErrors)
        {
            if (sslPolicyErrors == SslPolicyErrors.None)
                return true;

            Console.WriteLine("Certificate error: {0}", sslPolicyErrors);

            return false;
        }

        public override async Task<bool> CheckResponse(int expectedCode, CancellationToken cancellationToken)
        {
            if (_sslStream == null)
            {
                return false;
            }
            var message = await ReadMessageFromStream(_sslStream, cancellationToken);
            int responseCode = Convert.ToInt32(message.Substring(0, 3));
            if (responseCode == expectedCode)
            {
                return true;
            }
            return false;
        }

        public override void SendData(string data)
        {
            byte[] messsage = Encoding.UTF8.GetBytes(data);
            _sslStream.Write(messsage);
            _sslStream.Flush();
        }

        private async Task<string> ReadMessageFromStream(SslStream stream,CancellationToken cancellationToken)
        {
            byte[] buffer = new byte[2048];
            StringBuilder messageData = new StringBuilder();
            int bytes = -1;
            do
            {
                bytes =await stream.ReadAsync(buffer, 0, buffer.Length, cancellationToken);

                Decoder decoder = Encoding.UTF8.GetDecoder();
                char[] chars = new char[decoder.GetCharCount(buffer, 0, bytes)];
                decoder.GetChars(buffer, 0, bytes, chars, 0);
                messageData.Append(chars);
                if (messageData.ToString().IndexOf(EOF) != -1)
                {
                    break;
                }
            } while (bytes != 0);

            return messageData.ToString();
        }
    }
}
